(function initAutoHDProStorage() {
  if (globalThis.AutoHDProStorage) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = globalThis.AutoHDProStorage;
    }
    return;
  }

  /**
   * AutoHDPro Storage Utility
   * Manages extension settings and per-site configurations
   */

  const DEFAULT_SETTINGS = {
    enabled: true,
    globalCaptionsOff: true,
    autoQuality: true,
    removeShorts: false,
    preferredQuality: 'highest', // highest, 2160p, 1440p, 1080p, 720p
    siteOverrides: {},
    debugMode: false,
    statistics: {
      videosEnhanced: 0,
      totalTimeSaved: 0
    },
    siteStats: {}, // Per-site video counts
    subscription: {
      plan: 'free',
      monthlyConversions: 0,
      monthlyLimit: 1000,
      resetDate: null,
      userId: null
    }
  };

  const SITE_OVERRIDE_DEFAULTS = {
    enabled: true,
    captionsOff: true,
    quality: 'highest'
  };

  class StorageManager {
    constructor() {
      this.cache = null;
      this.listeners = new Set();
    }

    /**
     * Initialize storage with defaults
     */
    async init() {
      // Check if chrome.storage is available
      if (typeof chrome === 'undefined' || !chrome.storage) {
        console.warn('AutoHDPro: chrome.storage not available, using defaults');
        this.cache = { ...DEFAULT_SETTINGS };
        return this.cache;
      }

      try {
        const stored = await chrome.storage.local.get('settings');
        
        if (!stored.settings) {
          await chrome.storage.local.set({ settings: DEFAULT_SETTINGS });
          this.cache = { ...DEFAULT_SETTINGS };
        } else {
          // Merge with defaults to ensure all keys exist
          this.cache = { ...DEFAULT_SETTINGS, ...stored.settings };
        }
        
        // Listen for changes from other contexts
        chrome.storage.onChanged.addListener(this._handleStorageChange.bind(this));
        
        return this.cache;
      } catch (error) {
        console.error('AutoHDPro: Storage initialization failed:', error);
        this.cache = { ...DEFAULT_SETTINGS };
        return this.cache;
      }
    }

    /**
     * Get all settings
     */
    async getSettings() {
      if (!this.cache) {
        await this.init();
      }
      return { ...this.cache };
    }

    /**
     * Get specific setting
     */
    async get(key) {
      const settings = await this.getSettings();
      return key.includes('.') 
        ? this._getNestedValue(settings, key)
        : settings[key];
    }

    /**
     * Update settings
     */
    async set(updates) {
      const settings = await this.getSettings();
      
      // Handle nested updates
      Object.entries(updates).forEach(([key, value]) => {
        if (key.includes('.')) {
          this._setNestedValue(settings, key, value);
        } else {
          settings[key] = value;
        }
      });
      
      this.cache = settings;
      
      // Only try to save if chrome.storage is available
      if (typeof chrome !== 'undefined' && chrome.storage) {
        try {
          await chrome.storage.local.set({ settings });
        } catch (error) {
          console.error('AutoHDPro: Failed to save settings:', error);
        }
      }
      
      this._notifyListeners(updates);
      return settings;
    }

    /**
     * Get site-specific override
     */
    async getSiteOverride(hostname) {
      const settings = await this.getSettings();
      const overrides = settings.siteOverrides || {};
      
      // Check exact match first
      if (overrides[hostname]) {
        return { ...SITE_OVERRIDE_DEFAULTS, ...overrides[hostname] };
      }
      
      // Check wildcard matches
      for (const [pattern, override] of Object.entries(overrides)) {
        if (pattern.includes('*')) {
          const regex = new RegExp(pattern.replace(/\*/g, '.*'));
          if (regex.test(hostname)) {
            return { ...SITE_OVERRIDE_DEFAULTS, ...override };
          }
        }
      }
      
      return null;
    }

    /**
     * Set site-specific override
     */
    async setSiteOverride(hostname, override) {
      const settings = await this.getSettings();
      
      if (!settings.siteOverrides) {
        settings.siteOverrides = {};
      }
      
      if (override === null) {
        // Remove override
        delete settings.siteOverrides[hostname];
      } else {
        settings.siteOverrides[hostname] = {
          ...SITE_OVERRIDE_DEFAULTS,
          ...override
        };
      }
      
      await this.set({ siteOverrides: settings.siteOverrides });
    }

    /**
     * Update statistics
     */
    async updateStats(stats) {
      const settings = await this.getSettings();
      const currentStats = settings.statistics || {};
      
      Object.entries(stats).forEach(([key, value]) => {
        if (typeof value === 'number') {
          currentStats[key] = (currentStats[key] || 0) + value;
        } else {
          currentStats[key] = value;
        }
      });
      
      const updatedSettings = await this.set({ statistics: currentStats });
      return updatedSettings.statistics;
    }

    /**
     * Get video count for a specific site
     */
    async getSiteVideoCount(hostname) {
      const settings = await this.getSettings();
      const siteStats = settings.siteStats || {};
      return siteStats[hostname] || 0;
    }

    /**
     * Update video count for a specific site
     */
    async updateSiteVideoCount(hostname, increment = 1) {
      const settings = await this.getSettings();
      const siteStats = settings.siteStats || {};
      siteStats[hostname] = (siteStats[hostname] || 0) + increment;
      
      const updatedSettings = await this.set({ siteStats });
      return updatedSettings.siteStats[hostname];
    }

    /**
     * Reset video count for a specific site
     */
    async resetSiteVideoCount(hostname) {
      const settings = await this.getSettings();
      const siteStats = settings.siteStats || {};
      siteStats[hostname] = 0;
      
      const updatedSettings = await this.set({ siteStats });
      return updatedSettings.siteStats[hostname];
    }

    /**
     * Reset to defaults
     */
    async reset() {
      this.cache = { ...DEFAULT_SETTINGS };
      // Only persist if chrome.storage is available
      if (typeof chrome !== 'undefined' && chrome.storage) {
        try {
          await chrome.storage.local.set({ settings: DEFAULT_SETTINGS });
        } catch (error) {
          console.error('AutoHDPro: Failed to reset settings:', error);
        }
      }
      this._notifyListeners({ reset: true });
    }

    /**
     * Add change listener
     */
    onChange(callback) {
      this.listeners.add(callback);
      return () => this.listeners.delete(callback);
    }

    /**
     * Private: Handle storage changes
     */
    _handleStorageChange(changes, area) {
      if (area === 'local' && changes.settings) {
        this.cache = changes.settings.newValue;
        this._notifyListeners({ external: true, ...changes.settings.newValue });
      }
    }

    /**
     * Private: Notify listeners
     */
    _notifyListeners(changes) {
      this.listeners.forEach(callback => {
        try {
          callback(changes);
        } catch (e) {
          console.error('AutoHDPro: Storage listener error:', e);
        }
      });
    }

    /**
     * Private: Get nested value
     */
    _getNestedValue(obj, path) {
      return path.split('.').reduce((current, key) => current?.[key], obj);
    }

    /**
     * Private: Set nested value
     */
    _setNestedValue(obj, path, value) {
      const keys = path.split('.');
      const lastKey = keys.pop();
      const target = keys.reduce((current, key) => {
        if (!current[key]) current[key] = {};
        return current[key];
      }, obj);
      target[lastKey] = value;
    }
  }

  // Create singleton instance lazily
  let storageInstance = null;

  function getStorage() {
    if (!storageInstance) {
      storageInstance = new StorageManager();
    }
    return storageInstance;
  }

  // Export for use in all extension contexts
  const AutoHDProStorageExport = {
    async init() {
      return getStorage().init();
    },
    async getSettings() {
      return getStorage().getSettings();
    },
    async get(key) {
      return getStorage().get(key);
    },
    async set(updates) {
      return getStorage().set(updates);
    },
    async getSiteOverride(hostname) {
      return getStorage().getSiteOverride(hostname);
    },
    async setSiteOverride(hostname, override) {
      return getStorage().setSiteOverride(hostname, override);
    },
    async updateStats(stats) {
      return getStorage().updateStats(stats);
    },
    async getSiteVideoCount(hostname) {
      return getStorage().getSiteVideoCount(hostname);
    },
    async updateSiteVideoCount(hostname, increment) {
      return getStorage().updateSiteVideoCount(hostname, increment);
    },
    async resetSiteVideoCount(hostname) {
      return getStorage().resetSiteVideoCount(hostname);
    },
    async reset() {
      return getStorage().reset();
    },
    onChange(callback) {
      return getStorage().onChange(callback);
    }
  };

  try {
    // globalThis works in both window and worker contexts
    globalThis.AutoHDProStorage = AutoHDProStorageExport;
  } catch (_) {
    // Fallbacks for very old runtimes
    if (typeof self !== 'undefined') self.AutoHDProStorage = AutoHDProStorageExport;
    if (typeof window !== 'undefined') window.AutoHDProStorage = AutoHDProStorageExport;
  }

  if (typeof module !== 'undefined' && module.exports) {
    module.exports = AutoHDProStorageExport;
  }
})();
