/**
 * AutoHDPro Logging Utility
 * Provides consistent logging across the extension with debug levels
 */

(() => {
  const LOG_LEVELS = {
    ERROR: 0,
    WARN: 1,
    INFO: 2,
    DEBUG: 3
  };

  class Logger {
    constructor(module = 'AutoHDPro') {
      this.module = module;
      this.level = LOG_LEVELS.INFO;
      this.enabled = true;
    }

    setLevel(level) {
      this.level = typeof level === 'string' ? LOG_LEVELS[level.toUpperCase()] : level;
    }

    disable() {
      this.enabled = false;
    }

    enable() {
      this.enabled = true;
    }

    _log(level, message, ...args) {
      if (!this.enabled || level > this.level) return;

      const prefix = `AutoHDPro: [${this.module}]`;
      const timestamp = new Date().toISOString();

      switch (level) {
        case LOG_LEVELS.ERROR:
          console.error(`${prefix}[ERROR][${timestamp}]`, message, ...args);
          break;
        case LOG_LEVELS.WARN:
          console.warn(`${prefix}[WARN][${timestamp}]`, message, ...args);
          break;
        case LOG_LEVELS.INFO:
          console.info(`${prefix}[INFO][${timestamp}]`, message, ...args);
          break;
        case LOG_LEVELS.DEBUG:
          console.log(`${prefix}[DEBUG][${timestamp}]`, message, ...args);
          break;
      }
    }

    error(message, ...args) {
      this._log(LOG_LEVELS.ERROR, message, ...args);
    }

    warn(message, ...args) {
      this._log(LOG_LEVELS.WARN, message, ...args);
    }

    info(message, ...args) {
      this._log(LOG_LEVELS.INFO, message, ...args);
    }

    debug(message, ...args) {
      this._log(LOG_LEVELS.DEBUG, message, ...args);
    }

    group(label) {
      if (this.enabled) {
        console.group(`AutoHDPro: [${this.module}] ${label}`);
      }
    }

    groupEnd() {
      if (this.enabled) {
        console.groupEnd();
      }
    }

    time(label) {
      if (this.enabled && this.level >= LOG_LEVELS.DEBUG) {
        console.time(`AutoHDPro: [${this.module}] ${label}`);
      }
    }

    timeEnd(label) {
      if (this.enabled && this.level >= LOG_LEVELS.DEBUG) {
        console.timeEnd(`AutoHDPro: [${this.module}] ${label}`);
      }
    }
  }

  // Export factory function for creating module-specific loggers
  try {
    globalThis.AutoHDProLogger = Logger;
  } catch (_) {
    if (typeof self !== 'undefined') self.AutoHDProLogger = Logger;
    if (typeof window !== 'undefined') window.AutoHDProLogger = Logger;
  }

  if (typeof module !== 'undefined' && module.exports) {
    module.exports = Logger;
  }
})();
