(function initDOMUtils() {
  if (globalThis.AutoHDProDOM) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = globalThis.AutoHDProDOM;
    }
    return;
  }

  /**
   * AutoHDPro DOM Utilities
   * Helper functions for DOM manipulation and observation
   */

  const DOMUtils = {
    /**
     * Wait for an element to appear in the DOM
     */
    waitForElement(selector, timeout = 10000, root = document) {
      return new Promise((resolve, reject) => {
        const element = root.querySelector(selector);
        if (element) {
          resolve(element);
          return;
        }

        const observer = new MutationObserver((mutations, obs) => {
          const element = root.querySelector(selector);
          if (element) {
            obs.disconnect();
            resolve(element);
          }
        });

        observer.observe(root, {
          childList: true,
          subtree: true
        });

        if (timeout > 0) {
          setTimeout(() => {
            observer.disconnect();
            reject(new Error(`Timeout waiting for element: ${selector}`));
          }, timeout);
        }
      });
    },

    /**
     * Wait for multiple elements
     */
    waitForElements(selectors, timeout = 10000, root = document) {
      const promises = selectors.map(selector => 
        this.waitForElement(selector, timeout, root)
      );
      return Promise.allSettled(promises);
    },

    /**
     * Observe DOM changes with debouncing
     */
    observeChanges(element, callback, options = {}, debounceMs = 250) {
      let debounceTimer;
      
      const observer = new MutationObserver((mutations) => {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
          callback(mutations);
        }, debounceMs);
      });

      const defaultOptions = {
        childList: true,
        subtree: true,
        attributes: false,
        attributeOldValue: false,
        characterData: false,
        characterDataOldValue: false
    };

    observer.observe(element, { ...defaultOptions, ...options });
    return observer;
  },

  /**
   * Find all video elements including in iframes
   */
  findAllVideos(root = document) {
    const videos = [...root.querySelectorAll('video')];
    
    // Check iframes
    const iframes = root.querySelectorAll('iframe');
    iframes.forEach(iframe => {
      try {
        if (iframe.contentDocument) {
          videos.push(...iframe.contentDocument.querySelectorAll('video'));
        }
      } catch (e) {
        // Cross-origin iframe, skip
      }
    });
    
    return videos;
  },

  /**
   * Inject script into the page context
   */
  injectScript(code, removeAfter = true) {
    const script = document.createElement('script');
    script.textContent = code;
    (document.head || document.documentElement).appendChild(script);
    
    if (removeAfter) {
      script.remove();
    }
    
    return script;
  },

  /**
   * Inject script file into the page
   */
  injectScriptFile(src, removeAfter = false) {
    return new Promise((resolve, reject) => {
      const script = document.createElement('script');
      script.src = chrome.runtime.getURL(src);
      
      script.onload = () => {
        if (removeAfter) script.remove();
        resolve(script);
      };
      
      script.onerror = reject;
      
      (document.head || document.documentElement).appendChild(script);
    });
  },

  /**
   * Check if element is visible
   */
  isVisible(element) {
    if (!element) return false;
    
    const style = window.getComputedStyle(element);
    if (style.display === 'none' || style.visibility === 'hidden') {
      return false;
    }
    
    const rect = element.getBoundingClientRect();
    return rect.width > 0 && rect.height > 0;
  },

  /**
   * Simulate click on element
   */
  simulateClick(element) {
    const event = new MouseEvent('click', {
      view: window,
      bubbles: true,
      cancelable: true
    });
    element.dispatchEvent(event);
  },

  /**
   * Get closest parent matching selector
   */
  getClosest(element, selector, maxDepth = 10) {
    let parent = element.parentElement;
    let depth = 0;
    
    while (parent && depth < maxDepth) {
      if (parent.matches(selector)) {
        return parent;
      }
      parent = parent.parentElement;
      depth++;
    }
    
    return null;
  },

  /**
   * Create element from HTML string
   */
  createElement(html) {
    const template = document.createElement('template');
    template.innerHTML = html.trim();
    return template.content.firstChild;
  },

  /**
   * Safely query selector with error handling
   */
  safeQuerySelector(selector, root = document) {
    try {
      return root.querySelector(selector);
    } catch (e) {
      console.error('AutoHDPro: Invalid selector:', selector, e);
      return null;
    }
  },

  /**
   * Add CSS to page
   */
  addStyles(css, id = null) {
    const style = document.createElement('style');
    if (id) style.id = id;
    style.textContent = css;
    (document.head || document.documentElement).appendChild(style);
    return style;
  }
  };

  // Export for use in content scripts and workers
  try {
    globalThis.AutoHDProDOM = DOMUtils;
  } catch (_) {
    if (typeof self !== 'undefined') self.AutoHDProDOM = DOMUtils;
    if (typeof window !== 'undefined') window.AutoHDProDOM = DOMUtils;
  }

  if (typeof module !== 'undefined' && module.exports) {
    module.exports = DOMUtils;
  }
})();
