/**
 * AutoHDPro Popup Script
 */

(async function() {
  'use strict';
  
  // Initialize storage
  await AutoHDProStorage.init();
  
  // DOM elements
  const elements = {
    enabled: document.getElementById('enabled'),
    captionsOff: document.getElementById('captionsOff'),
    autoQuality: document.getElementById('autoQuality'),
    quality: document.getElementById('quality'),
    removeShorts: document.getElementById('removeShorts'),
    removeShortsHint: document.getElementById('removeShortsHint'),
    currentVideos: document.getElementById('currentVideos'),
    videosEnhanced: document.getElementById('videosEnhanced'),
    siteOverride: document.getElementById('siteOverride'),
    about: document.getElementById('about'),
    support: document.getElementById('support')
  };

  const SHORTS_SUPPORTED_HOSTS = [
    /youtube\.com/i,
    /youtu\.be/i
  ];

  let currentSettings = null;
  let supportsShortsRemoval = false;
  let activeTabId = null;

  function updateRemoveShortsButton() {
    const enabled = Boolean(currentSettings?.removeShorts);
    const button = elements.removeShorts;

    if (!button) {
      return;
    }

    button.dataset.active = enabled ? 'true' : 'false';
    button.textContent = enabled ? 'Show Shorts' : 'Remove Shorts';
    button.disabled = !supportsShortsRemoval;

    if (elements.removeShortsHint) {
      elements.removeShortsHint.style.opacity = supportsShortsRemoval ? '0.6' : '0.35';
    }
  }

  const aboutUrl = (window.AUTOHDPRO_CONFIG?.ABOUT_URL || '').trim() || 'https://autohdpro.com/about';
  const supportUrl = (window.AUTOHDPRO_CONFIG?.SUPPORT_URL || '').trim() || 'https://autohdpro.com/support';
  const extensionAboutUrl = chrome.runtime.getURL('src/ui/about.html');
  
  // Load current settings
  async function loadSettings() {
    const settings = await AutoHDProStorage.getSettings();
    currentSettings = settings;
    
    elements.enabled.checked = settings.enabled;
    elements.captionsOff.checked = settings.globalCaptionsOff;
    elements.autoQuality.checked = settings.autoQuality;
    elements.quality.value = settings.preferredQuality;
    elements.videosEnhanced.textContent = settings.statistics?.videosEnhanced || 0;

    updateRemoveShortsButton();
  }

  async function updateCurrentVideosDisplay(tabId) {
    if (!elements.currentVideos) {
      return;
    }

    if (!tabId) {
      elements.currentVideos.textContent = '0';
      return;
    }

    try {
      const response = await chrome.runtime.sendMessage({
        type: 'GET_TAB_STATS',
        tabId
      });

      const value = Number(response?.videosFound);
      elements.currentVideos.textContent = Number.isFinite(value) ? value : 0;
    } catch (error) {
      elements.currentVideos.textContent = '0';
    }
  }

  // Save setting
  async function saveSetting(key, value) {
    await chrome.runtime.sendMessage({
      type: 'UPDATE_SETTINGS',
      updates: { [key]: value }
    });
  }

  async function refreshActiveTabSettings() {
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab?.id) {
        await chrome.tabs.sendMessage(tab.id, { type: 'REFRESH_SETTINGS' });
      }
    } catch (error) {
      // No eligible content script on the active tab; ignore.
    }
  }
  
  // Event listeners
  elements.enabled.addEventListener('change', async (e) => {
    const enabled = e.target.checked;
    currentSettings = { ...currentSettings, enabled };
    await saveSetting('enabled', enabled);
    await refreshActiveTabSettings();
  });
  
  elements.captionsOff.addEventListener('change', (e) => {
    saveSetting('globalCaptionsOff', e.target.checked);
  });
  
  elements.autoQuality.addEventListener('change', async (e) => {
    const enabled = e.target.checked;
    await saveSetting('autoQuality', enabled);
    currentSettings = { ...currentSettings, autoQuality: enabled };
    await refreshActiveTabSettings();
  });
  
  elements.quality.addEventListener('change', async (e) => {
    const value = e.target.value;
    await saveSetting('preferredQuality', value);
    currentSettings = { ...currentSettings, preferredQuality: value };
    await refreshActiveTabSettings();
  });

  elements.removeShorts.addEventListener('click', async () => {
    if (!supportsShortsRemoval) {
      return;
    }

    const newValue = !currentSettings?.removeShorts;
    currentSettings = {
      ...currentSettings,
      removeShorts: newValue
    };

    updateRemoveShortsButton();
    await saveSetting('removeShorts', newValue);

    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab?.id) {
        await chrome.tabs.sendMessage(tab.id, {
          type: 'SET_REMOVE_SHORTS',
          enabled: newValue
        });
      }
    } catch (error) {
      // Most likely no content script on this tab; ignore
    }
  });

  elements.siteOverride.addEventListener('click', async () => {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (tab?.url) {
      const url = new URL(tab.url);
      const hostname = url.hostname;
      
      // Get current override
      const response = await chrome.runtime.sendMessage({
        type: 'GET_SITE_OVERRIDE',
        hostname
      });
      
      const override = response.override || { enabled: true, captionsOff: true, quality: 'highest' };
      
      // Simple toggle for demo
      const newEnabled = !override.enabled;
      
      await chrome.runtime.sendMessage({
        type: 'SET_SITE_OVERRIDE',
        hostname,
        override: newEnabled ? null : { enabled: false }
      });
      
      elements.siteOverride.textContent = newEnabled ? 'ON' : 'OFF';
      await refreshActiveTabSettings();
    }
  });
  elements.about.addEventListener('click', (e) => {
    e.preventDefault();

    const openExternal = e.metaKey || e.ctrlKey || e.shiftKey;
    const targetUrl = openExternal ? aboutUrl : extensionAboutUrl;
    chrome.tabs.create({ url: targetUrl });
  });
  
  elements.support.addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: supportUrl });
  });
  
  // Check current tab
  async function checkCurrentTab() {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    activeTabId = tab?.id ?? null;
    await updateCurrentVideosDisplay(activeTabId);

    if (tab?.url) {
      try {
        const url = new URL(tab.url);
        supportsShortsRemoval = SHORTS_SUPPORTED_HOSTS.some((pattern) => pattern.test(url.hostname));
        updateRemoveShortsButton();
        const response = await chrome.runtime.sendMessage({
          type: 'GET_SITE_OVERRIDE',
          hostname: url.hostname
        });
        
        if (response.override && !response.override.enabled) {
          elements.siteOverride.textContent = 'OFF';
        }
      } catch (e) {
        // Invalid URL
        supportsShortsRemoval = false;
        updateRemoveShortsButton();
      }
    } else {
      supportsShortsRemoval = false;
      updateRemoveShortsButton();
    }
  }

  // Initialize
  await loadSettings();
  await checkCurrentTab();

  // Listen for storage changes
  AutoHDProStorage.onChange(async () => {
    await loadSettings();
    await updateCurrentVideosDisplay(activeTabId);
  });

  chrome.runtime.onMessage.addListener((message) => {
    if (message?.type === 'TAB_STATS_UPDATED' && typeof message.videosFound !== 'undefined') {
      if (message.tabId && message.tabId === activeTabId) {
        const value = Number(message.videosFound);
        elements.currentVideos.textContent = Number.isFinite(value) ? value : 0;
      }
    }
  });
  
})();
