(async function() {
  'use strict';

  const siteListContainer = document.getElementById('siteList');
  const statusElement = document.getElementById('status');
  const searchInput = document.getElementById('siteSearch');
  const categoryTemplate = document.getElementById('categoryTemplate');
  const siteTemplate = document.getElementById('siteTemplate');
  const backButton = document.getElementById('openPopup');

  const dataUrl = chrome.runtime.getURL('src/data/supported-sites.json');
  const stateBySiteId = new Map();

  let categories = [];
  let totalSiteCount = 0;
  let currentFilter = '';
  const MAX_DATA_ATTEMPTS = 3;
  const RETRY_DELAY_MS = 800;

  const wait = (ms) => new Promise(resolve => setTimeout(resolve, ms));

  async function loadSupportedSitesData() {
    let lastError = null;

    for (let attempt = 1; attempt <= MAX_DATA_ATTEMPTS; attempt++) {
      try {
        const response = await fetch(dataUrl);
        if (!response.ok) {
          throw new Error(`Failed to load data: ${response.status}`);
        }
        return await response.json();
      } catch (error) {
        lastError = error;

        if (attempt < MAX_DATA_ATTEMPTS) {
          console.warn('AutoHDPro: supported sites fetch failed', {
            attempt,
            maxAttempts: MAX_DATA_ATTEMPTS,
            error
          });
          showStatus('Retrying to load supported sites...');
          await wait(RETRY_DELAY_MS * attempt);
        }
      }
    }

    throw lastError;
  }

  if (typeof AutoHDProStorage?.init === 'function') {
    try {
      await AutoHDProStorage.init();
    } catch (error) {
      console.warn('AutoHDPro: failed to initialize storage on about page', error);
    }
  }

  let sitesData;
  try {
    sitesData = await loadSupportedSitesData();
  } catch (error) {
    console.error('AutoHDPro: unable to load supported sites data after retries', error);
    showStatus('Could not load supported sites after multiple attempts. Please try again later.', true);
    return;
  }

  categories = Array.isArray(sitesData?.categories) ? sitesData.categories : [];

  totalSiteCount = categories.reduce((acc, category) => acc + (category?.sites?.length || 0), 0);

  async function resolveInitialStates() {
    for (const category of categories) {
      if (!Array.isArray(category?.sites)) {
        continue;
      }
      for (const site of category.sites) {
        if (!site?.id) {
          continue;
        }
        try {
          const enabled = await getSiteEnabled(site);
          stateBySiteId.set(site.id, enabled);
        } catch (error) {
          console.warn('AutoHDPro: failed to resolve site state', site, error);
          stateBySiteId.set(site.id, true);
        }
      }
    }
  }

  async function getSiteEnabled(site) {
    if (!Array.isArray(site?.domains) || site.domains.length === 0) {
      return true;
    }

    for (const domain of site.domains) {
      try {
        const response = await chrome.runtime.sendMessage({
          type: 'GET_SITE_OVERRIDE',
          hostname: domain
        });
        if (response?.override && response.override.enabled === false) {
          return false;
        }
      } catch (error) {
        console.warn('AutoHDPro: override lookup failed for domain', domain, error);
      }
    }

    return true;
  }

  function createSiteCard(categoryId, site) {
    const fragment = siteTemplate.content.cloneNode(true);
    const card = fragment.querySelector('[data-site-card]');
    const nameEl = fragment.querySelector('[data-site-name]');
    const domainsEl = fragment.querySelector('[data-site-domains]');
    const notesEl = fragment.querySelector('[data-site-notes]');
    const toggle = fragment.querySelector('[data-site-toggle]');

    nameEl.textContent = site.name;
    domainsEl.textContent = site.domains.join(', ');
    notesEl.textContent = site.notes || '';

    const enabled = stateBySiteId.get(site.id);
    toggle.checked = enabled !== false;

    toggle.addEventListener('change', async (event) => {
      const shouldEnable = event.target.checked;
      toggle.disabled = true;

      try {
        await setSiteEnabled(site, shouldEnable);
        stateBySiteId.set(site.id, shouldEnable);
        showStatus(`${site.name} ${shouldEnable ? 'enabled' : 'disabled'} for AutoHDPro.`);
      } catch (error) {
        console.error('AutoHDPro: failed to toggle site override', error);
        event.target.checked = !shouldEnable;
        showStatus(`Could not update ${site.name}. Please try again.`, true);
      } finally {
        toggle.disabled = false;
      }
    });

    return card;
  }

  async function setSiteEnabled(site, enabled) {
    if (!Array.isArray(site?.domains) || site.domains.length === 0) {
      return;
    }

    for (const domain of site.domains) {
      await chrome.runtime.sendMessage({
        type: 'SET_SITE_OVERRIDE',
        hostname: domain,
        override: enabled ? null : { enabled: false }
      });
    }
  }

  function render(filter) {
    const normalizedFilter = (filter || '').trim().toLowerCase();
    siteListContainer.innerHTML = '';

    let visibleCount = 0;

    for (const category of categories) {
      const sites = Array.isArray(category?.sites) ? category.sites : [];
      const matches = sites.filter((site) => {
        if (!normalizedFilter) {
          return true;
        }
        const haystack = [
          site.name,
          ...(site.domains || []),
          category.label || ''
        ]
          .join(' ')
          .toLowerCase();
        return haystack.includes(normalizedFilter);
      });

      if (matches.length === 0) {
        continue;
      }

      const categoryFragment = categoryTemplate.content.cloneNode(true);
      const categoryEl = categoryFragment.querySelector('[data-category]');
      const nameEl = categoryFragment.querySelector('[data-category-name]');
      const descriptionEl = categoryFragment.querySelector('[data-category-description]');
      const gridEl = categoryFragment.querySelector('[data-site-grid]');

      nameEl.textContent = category.label;
      descriptionEl.textContent = category.description || '';

      for (const site of matches) {
        const card = createSiteCard(category.id, site);
        gridEl.appendChild(card);
        visibleCount += 1;
      }

      siteListContainer.appendChild(categoryEl);
    }

    if (visibleCount === 0) {
      siteListContainer.innerHTML = '<p class="empty">No sites match your search yet.</p>';
    }

    if (!normalizedFilter) {
      showStatus(`Showing ${visibleCount} supported sites.`);
    } else {
      showStatus(`Showing ${visibleCount} of ${totalSiteCount} sites.`);
    }
  }

  function showStatus(message, isError = false) {
    if (!statusElement) {
      return;
    }
    statusElement.textContent = message || '';
    statusElement.style.color = isError ? '#ff8585' : '#77d7ff';
  }

  searchInput?.addEventListener('input', (event) => {
    currentFilter = event.target.value;
    render(currentFilter);
  });

  backButton?.addEventListener('click', async (event) => {
    event.preventDefault();
    try {
      if (chrome.action?.openPopup) {
        await chrome.action.openPopup();
      }
    } catch (error) {
      console.warn('AutoHDPro: failed to open popup from about page', error);
    }

    // Give the popup a moment to open before closing this tab.
    setTimeout(() => window.close(), 120);
  });

  showStatus('Loading supported sites...');
  await resolveInitialStates();
  render(currentFilter);
})();
