// Site-specific hooks injected into page context
(function() {
  'use strict';

  const wait = (ms) => new Promise(resolve => setTimeout(resolve, ms));

  const isYouTube = window.location.hostname.includes('youtube.com') || window.location.hostname.includes('youtu.be');
  if (isYouTube) {
    const IGNORED_QUALITIES = new Set(['auto', 'default']);
    const QUALITY_HEIGHTS = {
      highres: 4320,
      hd2880: 2880,
      hd2160: 2160,
      hd1440: 1440,
      hd1080: 1080,
      hd960: 960,
      hd720: 720,
      large: 480,
      medium: 360,
      small: 240,
      tiny: 144
    };

    const getPlayer = () => document.querySelector('#movie_player');

    const getAvailableLevels = (player) => {
      if (!player || typeof player.getAvailableQualityLevels !== 'function') {
        return [];
      }
      const rawLevels = player.getAvailableQualityLevels() || [];
      return Array.from(new Set(
        rawLevels
          .map(level => (level || '').toLowerCase())
          .filter(level => level && !IGNORED_QUALITIES.has(level))
      ));
    };

    const applyQualityCandidate = (player, quality) => {
      if (!quality) {
        return false;
      }
      let applied = false;
      if (typeof player.setPlaybackQualitySelectionOverride === 'function') {
        try {
          player.setPlaybackQualitySelectionOverride(quality);
          applied = true;
        } catch (_) {}
      }
      if (typeof player.setPlaybackQualityRange === 'function') {
        try {
          player.setPlaybackQualityRange(quality, quality);
          applied = true;
        } catch (_) {}
      }
      if (typeof player.setPlaybackQuality === 'function') {
        try {
          player.setPlaybackQuality(quality);
          applied = true;
        } catch (_) {}
      }
      return applied;
    };

    const getCurrentQualityInfo = (player) => {
      if (!player) {
        return { label: null, height: 0 };
      }
      try {
        const label = typeof player.getPlaybackQuality === 'function' ? (player.getPlaybackQuality() || '').toLowerCase() : null;
        const height = QUALITY_HEIGHTS[label] || 0;
        return { label, height };
      } catch (_) {
        return { label: null, height: 0 };
      }
    };

    const handleSetQuality = async (payload) => {
      const player = getPlayer();
      if (!player) {
        return { success: false, error: 'player-missing' };
      }

      const {
        quality,
        fallbacks = [],
        attempts = 8,
        delayMs = 200
      } = payload || {};

      const candidates = Array.from(new Set([quality, ...fallbacks].filter(Boolean).map(q => q.toLowerCase())));
      if (!candidates.length) {
        return { success: false, error: 'no-candidates', currentQuality: getCurrentQualityInfo(player).label };
      }

      for (const candidate of candidates) {
        for (let i = 0; i < attempts; i++) {
          const didApply = applyQualityCandidate(player, candidate);
          await wait(delayMs);

          const { label, height } = getCurrentQualityInfo(player);
          const targetHeight = QUALITY_HEIGHTS[candidate] || 0;
          if (label === candidate || (didApply && height >= targetHeight && targetHeight > 0)) {
            return {
              success: true,
              appliedQuality: candidate,
              currentQuality: label,
              currentHeight: height
            };
          }
        }
      }

      const current = getCurrentQualityInfo(player);
      return {
        success: false,
        appliedQuality: null,
        currentQuality: current.label,
        currentHeight: current.height
      };
    };

    window.addEventListener('AutoHDPro:YouTubeCommand', async (event) => {
      const { action, payload, responseEvent } = event.detail || {};
      if (!responseEvent) {
        return;
      }

      let response;
      try {
        if (action === 'getAvailableQualityLevels') {
          const player = getPlayer();
          const levels = getAvailableLevels(player);
          response = {
            success: Boolean(levels.length),
            levels,
            hasPlayer: Boolean(player)
          };
        } else if (action === 'setPlaybackQuality') {
          response = await handleSetQuality(payload);
        } else if (action === 'getPlaybackQuality') {
          const player = getPlayer();
          const { label, height } = getCurrentQualityInfo(player);
          response = {
            success: Boolean(label),
            currentQuality: label,
            currentHeight: height
          };
        } else {
          response = { success: false, error: 'unknown-action' };
        }
      } catch (error) {
        response = {
          success: false,
          error: error?.message || 'command-error'
        };
      }

      window.dispatchEvent(new CustomEvent(responseEvent, { detail: response }));
    });
  }

  const isNetflix = window.location.hostname.includes('netflix.com');
  if (isNetflix) {
    const checkNetflix = setInterval(() => {
      if (window.netflix?.appContext?.state?.playerApp) {
        clearInterval(checkNetflix);
        console.log('AutoHDPro: Netflix player detected');
      }
    }, 1000);
  }
})();
