/**
 * AutoHDPro Shorts Blocker
 * Provides utilities for hiding short-form video shelves (e.g. YouTube Shorts).
 */

(function() {
  const ROOT_CLASS = 'autohdpro-hide-shorts';
  const STYLE_ID = 'autohdpro-shorts-style';
  const SHORT_ITEM_CLASS = 'autohdpro-short-item';
  const PROCESSED_ATTR = 'data-autohdpro-short-processed';
  const LINK_PROCESSED_ATTR = 'data-autohdpro-short-link';

  let styleElement = null;
  const observers = new Map();
  const readyListeners = new Map();
  let activeHandler = null;

  function ensureStyle(css) {
    if (styleElement) {
      return;
    }

    try {
      if (window.AutoHDProDOM?.addStyles) {
        styleElement = window.AutoHDProDOM.addStyles(css, STYLE_ID);
        return;
      }
    } catch (error) {
      console.warn('AutoHDPro: Failed using DOM helper for shorts style:', error);
    }

    const style = document.createElement('style');
    style.id = STYLE_ID;
    style.textContent = css;
    (document.head || document.documentElement).appendChild(style);
    styleElement = style;
  }

  function removeStyle() {
    if (styleElement) {
      styleElement.remove();
      styleElement = null;
    }
  }

  function collectMatches(root, selector) {
    const matches = [];

    if (!root) {
      return matches;
    }

    if (root.nodeType === 1 && root.matches(selector)) {
      matches.push(root);
    }

    if (typeof root.querySelectorAll === 'function') {
      root.querySelectorAll(selector).forEach((node) => matches.push(node));
    }

    return matches;
  }

  function markShortElement(element) {
    if (!element || element.hasAttribute(PROCESSED_ATTR)) {
      return;
    }

    element.classList.add(SHORT_ITEM_CLASS);
    element.setAttribute(PROCESSED_ATTR, 'true');
  }

  function markShortAnchor(anchor, containerSelectors) {
    if (!anchor || anchor.hasAttribute(LINK_PROCESSED_ATTR)) {
      return;
    }

    let container = null;
    for (const selector of containerSelectors) {
      const candidate = anchor.closest(selector);
      if (candidate) {
        container = candidate;
        break;
      }
    }

    markShortElement(container || anchor);
    anchor.setAttribute(LINK_PROCESSED_ATTR, 'true');
  }

  function clearMarks() {
    document.querySelectorAll(`.${SHORT_ITEM_CLASS}`).forEach((element) => {
      element.classList.remove(SHORT_ITEM_CLASS);
      element.removeAttribute(PROCESSED_ATTR);
    });

    document.querySelectorAll(`[${LINK_PROCESSED_ATTR}]`).forEach((element) => {
      element.removeAttribute(LINK_PROCESSED_ATTR);
    });
  }

  function observeWhenReady(key, callback) {
    if (observers.has(key)) {
      return observers.get(key);
    }

    const startObserver = () => {
      const target = document.body || document.documentElement;
      if (!target) {
        return;
      }

      const observer = new MutationObserver((mutations) => {
        for (const mutation of mutations) {
          mutation.addedNodes.forEach((node) => {
            if (node.nodeType === 1) {
              callback(node);
            }
          });
        }
      });

      observer.observe(target, {
        childList: true,
        subtree: true
      });

      observers.set(key, observer);
      readyListeners.delete(key);
      return observer;
    };

    if (document.body) {
      return startObserver();
    }

    const readyCallback = () => {
      startObserver();
      document.removeEventListener('DOMContentLoaded', readyCallback);
    };

    document.addEventListener('DOMContentLoaded', readyCallback);
    readyListeners.set(key, readyCallback);
    return null;
  }

  function disconnectObserver(key) {
    const observer = observers.get(key);
    if (observer) {
      observer.disconnect();
      observers.delete(key);
    }

    const listener = readyListeners.get(key);
    if (listener) {
      document.removeEventListener('DOMContentLoaded', listener);
      readyListeners.delete(key);
    }
  }

  // === YouTube implementation ===
  const YOUTUBE_CARD_CONTAINERS = [
    'ytd-rich-grid-media',
    'ytd-rich-item-renderer',
    'ytd-rich-grid-row',
    'ytd-grid-video-renderer',
    'ytd-video-renderer',
    'ytd-compact-video-renderer',
    'ytd-compact-media-item-renderer',
    'ytd-playlist-panel-video-renderer',
    'ytd-carousel-ad-renderer',
    'ytd-reel-item-renderer',
    'ytd-watch-next-secondary-results-renderer',
    'ytm-rich-item-renderer',
    'ytm-rich-grid-media-skeleton-renderer',
    'ytm-rich-grid-video-renderer',
    'ytm-rich-item-renderer-inline',
    'ytm-item-section-renderer',
    'ytm-video-with-context-renderer'
  ];

  const YOUTUBE_NAV_CONTAINERS = [
    'ytd-guide-entry-renderer',
    'ytd-mini-guide-entry-renderer',
    'ytd-guide-collapsible-entry-renderer',
    'ytm-pivot-bar-item-renderer',
    'ytm-channel-list-item-renderer',
    'yt-chip-cloud-chip-renderer',
    'yt-chip-renderer',
    'ytd-feed-filter-chip-bar-renderer',
    'ytm-pivot-bar-renderer'
  ];

  const YOUTUBE_STATIC_SELECTORS = [
    'ytd-rich-shelf-renderer[is-shorts]',
    'ytd-rich-shelf-renderer[is_shorts]',
    'ytd-reel-shelf-renderer',
    'ytd-reel-item-renderer',
    'ytd-reel-video-renderer',
    '#shorts-container',
    'ytd-reel-shelf-renderer',
    'ytm-shorts-shelf-renderer',
    'ytm-reel-shelf-renderer'
  ];

  const YOUTUBE_STYLE = `
    .${ROOT_CLASS} .${SHORT_ITEM_CLASS},
    .${ROOT_CLASS} ytd-rich-shelf-renderer[is-shorts],
    .${ROOT_CLASS} ytd-rich-shelf-renderer[is_shorts],
    .${ROOT_CLASS} ytd-reel-shelf-renderer,
    .${ROOT_CLASS} ytd-reel-item-renderer,
    .${ROOT_CLASS} ytd-reel-video-renderer,
    .${ROOT_CLASS} #shorts-container,
    .${ROOT_CLASS} ytm-shorts-shelf-renderer,
    .${ROOT_CLASS} ytm-reel-shelf-renderer,
    .${ROOT_CLASS} ytd-mini-guide-entry-renderer[aria-label*="Shorts"],
    .${ROOT_CLASS} ytd-guide-entry-renderer[title*="Shorts"],
    .${ROOT_CLASS} ytm-pivot-bar-item-renderer[tab-title*="Shorts"]
    { display: none !important; }
  `;

  function scanYouTube(node) {
    const root = node || document;

    YOUTUBE_STATIC_SELECTORS.forEach((selector) => {
      collectMatches(root, selector).forEach(markShortElement);
    });

    collectMatches(root, 'a[href*="/shorts/"]').forEach((anchor) => {
      markShortAnchor(anchor, YOUTUBE_CARD_CONTAINERS);
    });

    collectMatches(root, 'a[href^="/shorts"], a[title^="Shorts"], a[aria-label^="Shorts"]').forEach((anchor) => {
      markShortAnchor(anchor, YOUTUBE_NAV_CONTAINERS);
    });

    collectMatches(root, 'yt-chip-cloud-chip-renderer a[href*="/shorts"], ytd-feed-filter-chip-bar-renderer a[href*="/shorts"], ytm-pivot-bar-item-renderer a[href*="/shorts"]').forEach((anchor) => {
      markShortAnchor(anchor, YOUTUBE_NAV_CONTAINERS);
    });
  }

  function enableYouTube() {
    ensureStyle(YOUTUBE_STYLE);
    document.documentElement.classList.add(ROOT_CLASS);
    scanYouTube(document);
    observeWhenReady('youtube', scanYouTube);
  }

  function disableYouTube() {
    document.documentElement.classList.remove(ROOT_CLASS);
    disconnectObserver('youtube');
    clearMarks();
    removeStyle();
  }

  const handlers = [
    {
      id: 'youtube',
      matches: (location) => /youtube\.com|youtu\.be/i.test(location.hostname),
      enable: enableYouTube,
      disable: disableYouTube
    }
  ];

  function getHandler(location = window.location) {
    return handlers.find((handler) => {
      try {
        return handler.matches(location);
        } catch (error) {
          console.warn('AutoHDPro: Shorts handler match failed:', error);
        return false;
      }
    }) || null;
  }

  const ShortsBlocker = {
    enable() {
      const handler = getHandler();
      if (!handler) {
        return false;
      }

      if (activeHandler?.id === handler.id) {
        handler.enable();
        return true;
      }

      if (activeHandler) {
        activeHandler.disable();
      }

      handler.enable();
      activeHandler = handler;
      return true;
    },

    disable() {
      if (!activeHandler) {
        return false;
      }

      try {
        activeHandler.disable();
      } finally {
        activeHandler = null;
      }

      return true;
    },

    sync(enabled) {
      if (enabled) {
        return this.enable();
      }
      return this.disable();
    },

    isSupported(location = window.location) {
      return Boolean(getHandler(location));
    }
  };

  try {
    globalThis.AutoHDProShortsBlocker = ShortsBlocker;
  } catch (error) {
    window.AutoHDProShortsBlocker = ShortsBlocker;
  }
})();
