/**
 * AutoHDPro Quality Engine (safe syntax)
 * Manages video quality settings across different players
 */

(function(){
  'use strict';

  function QualityEngine() {
    this.logger = new AutoHDProLogger('Quality');
    this.qualityMap = {
      '2160p': ['2160p', '4K', 'UHD', '2160', '3840x2160'],
      '1440p': ['1440p', '2K', 'QHD', '1440', '2560x1440'],
      '1080p': ['1080p', 'FHD', 'Full HD', '1080', '1920x1080', 'hd1080'],
      '720p': ['720p', 'HD', '720', '1280x720', 'hd720'],
      '480p': ['480p', 'SD', '480', '854x480'],
      '360p': ['360p', '360', '640x360'],
      '240p': ['240p', '240', '426x240'],
      '144p': ['144p', '144', '256x144']
    };
    this.qualityOrder = ['2160p','1440p','1080p','720p','480p','360p','240p','144p'];
  }

  QualityEngine.prototype.setMaxQuality = function(video, options) {
    options = options || {};
    var driver = options.driver || null;
    var preferredQuality = options.preferredQuality || 'highest';
    this.logger.debug('Setting max quality for video', { preferredQuality: preferredQuality });

    var self = this;
    if (driver && typeof driver.setMaxQuality === 'function') {
      return Promise.resolve().then(function(){
        return driver.setMaxQuality(video, {
          preferredQuality: preferredQuality
        });
      }).then(function(result){
        if (result) {
          self.logger.info('Quality set via driver');
          return true;
        }
        return self.setQualityGeneric(video, preferredQuality);
      }).catch(function(){
        return self.setQualityGeneric(video, preferredQuality);
      });
    }
    return Promise.resolve(this.setQualityGeneric(video, preferredQuality));
  };

  QualityEngine.prototype.setQualityGeneric = function(video, preferredQuality) {
    var approaches = [
      this.tryVideoTracksAPI.bind(this, video, preferredQuality),
      this.tryMediaSourceAPI.bind(this, video, preferredQuality),
      this.tryQualityLevelsAPI.bind(this, video, preferredQuality),
      this.tryQualityMenuSelection.bind(this, video, preferredQuality)
    ];
    for (var i=0;i<approaches.length;i++) {
      try {
        if (approaches[i]()) {
          this.logger.info('Quality set using generic approach');
          return true;
        }
      } catch (e) {}
    }
    this.tryHTMLAttributes(video);
    this.logger.warn('Could not set quality - no compatible API found');
    return false;
  };

  QualityEngine.prototype.tryVideoTracksAPI = function(video, preferredQuality) {
    if (!video.videoTracks || video.videoTracks.length === 0) return false;
    var tracks = Array.prototype.slice.call(video.videoTracks);
    var targetTrack = this.selectBestTrack(tracks, preferredQuality);
    if (targetTrack) {
      tracks.forEach(function(t){ t.selected = false; });
      targetTrack.selected = true;
      this.logger.info('Selected video track: ' + (targetTrack.label || targetTrack.id));
      return true;
    }
    return false;
  };

  QualityEngine.prototype.tryMediaSourceAPI = function(video) {
    if (!video.src || video.src.indexOf('blob:') !== 0) return false;
    var playerAPIs = [video.player, video._player, video.videojs, video.plyr, video.jwplayer, video.flowplayer];
    for (var i=0;i<playerAPIs.length;i++) {
      var api = playerAPIs[i];
      if (!api) continue;
      var methods = [
        function(){ return api.setQuality && api.setQuality('highest'); },
        function(){ return api.setPlaybackQuality && api.setPlaybackQuality('highest'); },
        function(){ return api.setLevel && api.setLevel(-1); },
        function(){ if (api.quality){ api.quality = 'highest'; return true; } },
        function(){ return api.setQualityLevel && api.setQualityLevel(-1); }
      ];
      for (var j=0;j<methods.length;j++) {
        try { if (methods[j]()) { this.logger.info('Quality set via player API'); return true; } } catch(e) {}
      }
    }
    return false;
  };

  QualityEngine.prototype.tryQualityLevelsAPI = function(video) {
    var qualityLevels = video.qualityLevels || video._qualityLevels;
    if (!qualityLevels || typeof qualityLevels !== 'object') return false;
    var levels = [];
    for (var i=0;i<qualityLevels.length;i++) levels.push(qualityLevels[i]);
    if (!levels.length) return false;
    var highest = levels.reduce(function(best, cur){ return (!best || cur.height > best.height) ? cur : best; }, null);
    if (highest) {
      levels.forEach(function(l){ l.enabled = false; });
      highest.enabled = true;
      this.logger.info('Enabled quality level: ' + (highest.height || '') + 'p');
      return true;
    }
    return false;
  };

  QualityEngine.prototype.tryHTMLAttributes = function(video) {
    try {
      video.setAttribute('preload', 'auto');
      video.removeAttribute('bandwidth');
      video.removeAttribute('connectionSpeed');
      this.logger.debug('Applied HTML attribute hints for quality');
    } catch(_){ }
    return false;
  };

  QualityEngine.prototype.buildQualityHints = function(preferredQuality) {
    var hints = [];
    var addHint = function(value) {
      if (!value) return;
      var normalized = String(value).toLowerCase();
      if (hints.indexOf(normalized) === -1) {
        hints.push(normalized);
      }
    };

    var addQuality = function(quality) {
      if (!quality) return;
      addHint(quality);
      var aliases = this.qualityMap[quality];
      if (aliases && aliases.length) {
        aliases.forEach(function(alias) { addHint(alias); });
      }
    }.bind(this);

    if (preferredQuality && preferredQuality !== 'highest') {
      addQuality(preferredQuality);
    }

    this.qualityOrder.forEach(function(quality) {
      addQuality(quality);
    });

    ['4k', 'uhd', 'hd', 'full hd'].forEach(addHint);

    return hints;
  };

  QualityEngine.prototype.tryQualityMenuSelection = function(video, preferredQuality) {
    var doc = video.ownerDocument || document;
    if (!doc) return false;

    var hints = this.buildQualityHints(preferredQuality);
    if (!hints.length) return false;

    var self = this;

    var selectors = [
      '[data-quality]',
      '[data-quality-option]',
      '[data-quality-level]',
      '[data-quality-value]',
      '[data-select-quality]',
      '[data-action="quality"]',
      '[data-role="quality"]',
      '[aria-label*="quality" i]',
      '[class*="quality" i]',
      '[role="menuitem"]',
      'button',
      'li',
      'a',
      'div',
      'span'
    ];

    var uniqueElements = new Set();
    selectors.forEach(function(selector) {
      try {
        var nodes = doc.querySelectorAll(selector);
        for (var i = 0; i < nodes.length; i++) {
          uniqueElements.add(nodes[i]);
        }
      } catch (_) {}
    });

    if (uniqueElements.size === 0) {
      return false;
    }

    var elements = Array.from(uniqueElements);

    var normalizeDigits = function(input) {
      return String(input || '').replace(/[^0-9]/g, '');
    };

    var isVisible = function(element) {
      if (!element) return false;
      var rect = element.getBoundingClientRect();
      if (rect.width === 0 && rect.height === 0) return false;
      var style;
      try {
        style = doc.defaultView ? doc.defaultView.getComputedStyle(element) : window.getComputedStyle(element);
      } catch(_) {}
      if (style && (style.visibility === 'hidden' || style.display === 'none')) {
        return false;
      }
      return true;
    };

    var extractCombined = function(el) {
      var text = (el.textContent || '').toLowerCase();
      var aria = (el.getAttribute && el.getAttribute('aria-label')) ? el.getAttribute('aria-label').toLowerCase() : '';
      var dataQuality = '';
      if (el.dataset) {
        dataQuality = Object.keys(el.dataset).map(function(key) {
          if (!Object.prototype.hasOwnProperty.call(el.dataset, key)) return '';
          return String(el.dataset[key] || '').toLowerCase();
        }).join(' ');
      }
      return {
        element: el,
        text: text,
        combined: (text + ' ' + aria + ' ' + dataQuality).trim()
      };
    };

    var elementData = elements.map(extractCombined);

    var determineQualityFromSource = function(source) {
      if (!source) {
        return null;
      }
      var lower = String(source).toLowerCase();
      for (var qi = 0; qi < self.qualityOrder.length; qi++) {
        var qualityKey = self.qualityOrder[qi];
        var aliases = self.qualityMap[qualityKey] || [];
        var matched = false;
        for (var ai = 0; ai < aliases.length; ai++) {
          var aliasLower = String(aliases[ai]).toLowerCase();
          if (aliasLower && lower.indexOf(aliasLower) !== -1) {
            matched = true;
            break;
          }
        }
        if (!matched && lower.indexOf(String(qualityKey).toLowerCase()) !== -1) {
          matched = true;
        }
        if (matched) {
          return qualityKey;
        }
      }

      var digitMatches = lower.match(/(\d{3,4})/g);
      if (digitMatches) {
        for (var dm = 0; dm < digitMatches.length; dm++) {
          var candidate = digitMatches[dm] + 'p';
          if (self.qualityOrder.indexOf(candidate) !== -1) {
            return candidate;
          }
        }
      }
      return null;
    };

    var availableQualities = new Set();
    for (var ed = 0; ed < elementData.length; ed++) {
      var qualityKey = determineQualityFromSource(elementData[ed].combined);
      if (qualityKey) {
        availableQualities.add(qualityKey);
      }
    }

    if (availableQualities.size > 0) {
      var orderedQualities = [];

      var preferredKey = determineQualityFromSource(preferredQuality);
      if (preferredKey && availableQualities.has(preferredKey)) {
        orderedQualities.push(preferredKey);
      }

      for (var qo = 0; qo < self.qualityOrder.length; qo++) {
        var qKey = self.qualityOrder[qo];
        if (availableQualities.has(qKey) && orderedQualities.indexOf(qKey) === -1) {
          orderedQualities.push(qKey);
        }
      }

      var rebuiltHints = [];
      for (var oq = 0; oq < orderedQualities.length; oq++) {
        var base = orderedQualities[oq];
        var aliasList = [base].concat(self.qualityMap[base] || []);
        for (var al = 0; al < aliasList.length; al++) {
          var aliasValue = String(aliasList[al] || '').toLowerCase();
          if (aliasValue && rebuiltHints.indexOf(aliasValue) === -1) {
            rebuiltHints.push(aliasValue);
          }
        }
        var digitsValue = base.replace(/[^0-9]/g, '');
        if (digitsValue && rebuiltHints.indexOf(digitsValue) === -1) {
          rebuiltHints.push(digitsValue);
        }
      }

      if (rebuiltHints.length) {
        hints = rebuiltHints;
      }
    }

    for (var h = 0; h < hints.length; h++) {
      var hint = hints[h];
      var hintDigits = normalizeDigits(hint);

      for (var idx = 0; idx < elementData.length; idx++) {
        var data = elementData[idx];
        var el = data.element;

        if (!isVisible(el)) {
          continue;
        }

        var combined = data.combined;

        if (!combined) {
          continue;
        }

        if (combined.indexOf('auto') !== -1 && combined.indexOf(hint) === -1) {
          continue;
        }

        var matchesHint = combined.indexOf(hint) !== -1;

        if (!matchesHint && hintDigits) {
          var combinedDigits = normalizeDigits(combined);
          matchesHint = combinedDigits.indexOf(hintDigits) !== -1;
        }

        if (matchesHint) {
          try {
            el.dispatchEvent(new MouseEvent('click', { bubbles: true, cancelable: true }));
            if (typeof el.click === 'function') {
              el.click();
            }
            this.logger.info('Attempted quality selection via menu', {
              hint: hint,
              elementTag: el.tagName,
              text: data.text ? data.text.trim().slice(0, 60) : ''
            });
            return true;
          } catch (error) {
            this.logger.debug('Quality menu interaction failed', error);
          }
        }
      }
    }

    return false;
  };

  QualityEngine.prototype.selectBestTrack = function(tracks, preference) {
    if (preference === 'highest' || !preference) {
      return tracks.reduce(function(best, cur){
        var ch = (cur.label || cur.id || '').match(/(\d+)p?/);
        var bh = best ? (best.label || best.id || '').match(/(\d+)p?/) : null;
        var chn = ch ? parseInt(ch[1],10) : 0;
        var bhn = bh ? parseInt(bh[1],10) : 0;
        return chn > bhn ? cur : best;
      }, null);
    }
    var aliases = this.qualityMap[preference] || [preference];
    for (var i=0;i<aliases.length;i++){
      var alias = String(aliases[i]).toLowerCase();
      var found = tracks.find(function(t){
        var label = String(t.label || t.id || '').toLowerCase();
        return label.indexOf(alias) !== -1;
      });
      if (found) return found;
    }
    return this.selectBestTrack(tracks, 'highest');
  };

  QualityEngine.prototype.getCurrentQuality = function(video) {
    if (video.videoTracks && video.videoTracks.length > 0) {
      var sel = Array.prototype.slice.call(video.videoTracks).find(function(t){ return t.selected; });
      if (sel) return sel.label || sel.id;
    }
    var qualityLevels = video.qualityLevels || video._qualityLevels;
    if (qualityLevels) {
      for (var i=0;i<qualityLevels.length;i++) if (qualityLevels[i].enabled) return qualityLevels[i].label || (qualityLevels[i].height + 'p');
    }
    if (video.videoHeight) return String(video.videoHeight) + 'p';
    return 'unknown';
  };

  QualityEngine.prototype.forceQualityBySource = function(video, quality) {
    var src = video.src || video.currentSrc; if (!src) return Promise.resolve(false);
    var self=this;
    return Promise.resolve().then(function(){
      var url = new URL(src);
      url.searchParams.set('quality', quality);
      url.searchParams.set('q', quality);
      url.searchParams.set('vq', quality);
      var currentTime = video.currentTime; var wasPlaying = !video.paused;
      video.src = url.toString();
      video.addEventListener('loadedmetadata', function(){
        try { video.currentTime = currentTime; if (wasPlaying) video.play(); } catch(_){ }
      }, { once: true });
      self.logger.info('Forced quality via source reload');
      return true;
    }).catch(function(e){ self.logger.error('Failed to force quality by source:', e); return false; });
  };

  try { globalThis.AutoHDProQuality = new QualityEngine(); }
  catch(_){ if (typeof window!=='undefined') window.AutoHDProQuality = new QualityEngine(); }
  if (typeof module !== 'undefined' && module.exports) module.exports = QualityEngine;
})();
