(function initAutoHDProDetector() {
  const root = typeof globalThis !== 'undefined'
    ? globalThis
    : typeof self !== 'undefined'
      ? self
      : typeof window !== 'undefined'
        ? window
        : typeof global !== 'undefined'
          ? global
          : {};

  if (root.AutoHDProDetector) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = root.AutoHDProDetector.constructor;
    }
    return;
  }

  /**
   * AutoHDPro Video Detector
   * Detects and tracks video elements on the page
   */
  
  class VideoDetector {
    constructor() {
      this.logger = new AutoHDProLogger('Detector');
      this.videos = new Map();
      this.observer = null;
      this.callbacks = new Set();
      this.isObserving = false;
      this.mutationThrottle = null;
      this.scanThrottle = null;
      this.maxVideosPerPage = 10; // Limit videos per page
      this.scanCount = 0;
      this.maxScans = 5; // Maximum scans per page
    }
  
    /**
     * Start detecting videos
     */
    start() {
      if (this.isObserving) return;
      
      this.logger.info('Starting video detection');
      
      // Initial scan
      this.scanForVideos();
      
      // Set up mutation observer with throttling
      this.observer = new MutationObserver((mutations) => {
        this.throttledHandleMutations(mutations);
      });
      
      this.observer.observe(document.documentElement, {
        childList: true,
        subtree: true,
        attributes: true,
        attributeFilter: ['src', 'poster']
      });
      
      this.isObserving = true;
      
      // Listen for dynamically added videos
      this.setupVideoListeners();
      
      // Reduced periodic rescan - only every 10 seconds and only if no videos found
      this.rescanInterval = setInterval(() => {
        if (this.videos.size === 0) {
          this.scanForVideos();
        }
      }, 10000);
    }
  
    /**
     * Stop detecting videos
     */
    stop() {
      if (!this.isObserving) return;
      
      this.logger.info('Stopping video detection');
      
      if (this.observer) {
        this.observer.disconnect();
        this.observer = null;
      }
      
      if (this.rescanInterval) {
        clearInterval(this.rescanInterval);
        this.rescanInterval = null;
      }
      
      this.isObserving = false;
    }
  
    /**
     * Scan for video elements
     */
    scanForVideos() {
      // Limit scans per page
      if (this.scanCount >= this.maxScans) {
        this.logger.debug('Max scans reached, stopping detection');
        this.stop();
        return;
      }
      
      this.scanCount++;
      
      // Throttle scans
      if (this.scanThrottle) {
        clearTimeout(this.scanThrottle);
      }
      
      this.scanThrottle = setTimeout(() => {
        this.performScan();
      }, 100);
    }
  
    /**
     * Perform actual video scan
     */
    performScan() {
      // Limit videos per page
      if (this.videos.size >= this.maxVideosPerPage) {
        this.logger.debug('Max videos per page reached');
        return;
      }
  
      const videos = document.querySelectorAll('video');
      let newVideosFound = 0;
      
      videos.forEach(video => {
        if (!this.videos.has(video) && this.videos.size < this.maxVideosPerPage) {
          this.registerVideo(video);
          newVideosFound++;
        }
      });
      
      // Only scan containers if we haven't found many videos
      if (this.videos.size < 3) {
        this.scanForVideoContainers();
      }
      
      if (newVideosFound > 0) {
        this.logger.debug(`Found ${newVideosFound} new video(s), total: ${this.videos.size}`);
      }
    }
  
    /**
     * Scan for known video player containers
     */
    scanForVideoContainers() {
      const containerSelectors = [
        // YouTube
        '.html5-video-container',
        '#movie_player',
        'ytd-player',
        
        // Netflix
        '.VideoContainer',
        
        // Twitch
        '.video-player__container',
        
        // Vimeo
        '.vp-video-wrapper',
        
        // Generic players
        '.video-js',
        '.jwplayer',
        '.flowplayer',
        '.mejs-container',
        
        // Adult sites
        '.player-container',
        '.video-wrapper',
        '#player',
        '.mgvideo'
      ];
      
      containerSelectors.forEach(selector => {
        const containers = document.querySelectorAll(selector);
        containers.forEach(container => {
          const video = container.querySelector('video');
          if (video && !this.videos.has(video)) {
            this.registerVideo(video, container);
          }
        });
      });
    }
  
    /**
     * Register a video element
     */
    registerVideo(video, container = null) {
      if (this.videos.has(video)) return;
  
      const videoInfo = {
        element: video,
        container: container || video.parentElement,
        id: `video_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
        detected: new Date(),
        enhanced: false,
        processing: false,
        usageChecked: false,  // Track if we've already counted this video
        retryCount: 0,
        retryTimer: null,
        metadata: this.getVideoMetadata(video)
      };
      
      this.videos.set(video, videoInfo);
      
      // Set up video event listeners
      this.setupVideoEventListeners(video, videoInfo);
      
      // Notify callbacks
      this.notifyCallbacks('detected', videoInfo);
      
      this.logger.info(`Video detected: ${videoInfo.id}`, videoInfo.metadata);
    }
  
    /**
     * Get video metadata
     */
    getVideoMetadata(video) {
      return {
        src: video.src || video.currentSrc,
        duration: video.duration,
        width: video.videoWidth,
        height: video.videoHeight,
        poster: video.poster,
        autoplay: video.autoplay,
        muted: video.muted,
        controls: video.controls,
        readyState: video.readyState,
        networkState: video.networkState
      };
    }
  
    /**
     * Set up event listeners for a video
     */
    setupVideoEventListeners(video, videoInfo) {
      const events = [
        'loadstart',
        'loadedmetadata',
        'loadeddata',
        'canplay',
        'canplaythrough',
        'play',
        'playing',
        'pause',
        'ended',
        'error'
      ];
      
      events.forEach(eventName => {
        video.addEventListener(eventName, (event) => {
          this.handleVideoEvent(event, videoInfo);
        });
      });
      
      // Special handling for quality changes
      if ('videoTracks' in video) {
        video.videoTracks.addEventListener('change', () => {
          this.notifyCallbacks('qualitychange', videoInfo);
        });
      }
    }
  
    /**
     * Handle video events
     */
    handleVideoEvent(event, videoInfo) {
      const video = event.target;
      
      switch (event.type) {
        case 'loadedmetadata':
          // Update metadata
          videoInfo.metadata = this.getVideoMetadata(video);
          this.notifyCallbacks('metadata', videoInfo);
          break;
          
        case 'canplay':
        case 'canplaythrough':
          // Good time to enhance
          if (!videoInfo.enhanced) {
            this.notifyCallbacks('ready', videoInfo);
          }
          break;
          
        case 'play':
        case 'playing':
          this.notifyCallbacks('playing', videoInfo);
          break;
          
        case 'error':
          this.logger.error('Video error:', event);
          this.notifyCallbacks('error', videoInfo);
          break;
      }
    }
  
    /**
     * Throttled mutation handler
     */
    throttledHandleMutations(mutations) {
      if (this.mutationThrottle) {
        clearTimeout(this.mutationThrottle);
      }
      
      this.mutationThrottle = setTimeout(() => {
        this.handleMutations(mutations);
      }, 200); // 200ms throttle
    }
  
    /**
     * Handle DOM mutations
     */
    handleMutations(mutations) {
      // Skip if we've reached limits
      if (this.videos.size >= this.maxVideosPerPage || this.scanCount >= this.maxScans) {
        return;
      }
  
      let shouldScan = false;
      
      for (const mutation of mutations) {
        // Check for added nodes
        if (mutation.addedNodes.length > 0) {
          mutation.addedNodes.forEach(node => {
            if (node.nodeType === Node.ELEMENT_NODE) {
              if (node.tagName === 'VIDEO') {
                this.registerVideo(node);
              } else if (node.querySelector && this.videos.size < 3) {
                // Only check subtrees if we haven't found many videos
                const videos = node.querySelectorAll('video');
                videos.forEach(video => {
                  if (this.videos.size < this.maxVideosPerPage) {
                    this.registerVideo(video);
                  }
                });
              }
            }
          });
        }
        
        // Check for removed videos
        if (mutation.removedNodes.length > 0) {
          mutation.removedNodes.forEach(node => {
            if (node.nodeType === Node.ELEMENT_NODE && node.tagName === 'VIDEO') {
              this.unregisterVideo(node);
            }
          });
        }
        
        // Check for src changes
        if (mutation.type === 'attributes' && mutation.target.tagName === 'VIDEO') {
          const video = mutation.target;
          const videoInfo = this.videos.get(video);
          if (videoInfo) {
            const oldSrc = videoInfo.metadata.src;
            videoInfo.metadata = this.getVideoMetadata(video);
            // Only notify if src actually changed
            if (oldSrc !== videoInfo.metadata.src) {
              // Keep the usageChecked flag when src changes on same element
              this.notifyCallbacks('srcchange', videoInfo);
            }
          }
        }
      }
    }
  
    /**
     * Unregister a video
     */
    unregisterVideo(video) {
      const videoInfo = this.videos.get(video);
      if (videoInfo) {
        if (videoInfo.retryTimer) {
          clearTimeout(videoInfo.retryTimer);
          videoInfo.retryTimer = null;
        }
        this.videos.delete(video);
        this.notifyCallbacks('removed', videoInfo);
        this.logger.debug(`Video removed: ${videoInfo.id}`);
      }
    }
  
    /**
     * Set up global video listeners
     */
    setupVideoListeners() {
      // Override HTMLVideoElement.play to catch all plays
      const originalPlay = HTMLVideoElement.prototype.play;
      HTMLVideoElement.prototype.play = function() {
        const video = this;
        
        // Register if not already
        if (!window.AutoHDProDetector?.videos.has(video)) {
          window.AutoHDProDetector?.registerVideo(video);
        }
        
        return originalPlay.apply(this, arguments);
      };
    }
  
    /**
     * Register callback for video events
     */
    on(callback) {
      this.callbacks.add(callback);
      return () => this.callbacks.delete(callback);
    }
  
    /**
     * Notify all callbacks
     */
    notifyCallbacks(event, data) {
      this.callbacks.forEach(callback => {
        try {
          callback(event, data);
        } catch (e) {
          this.logger.error('Callback error:', e);
        }
      });
    }
  
    /**
     * Get all detected videos
     */
    getVideos() {
      return Array.from(this.videos.values());
    }
  
    /**
     * Get video info by element
     */
    getVideoInfo(video) {
      return this.videos.get(video);
    }
  
    /**
     * Mark video as enhanced
     */
    markEnhanced(video) {
      const videoInfo = this.videos.get(video);
      if (videoInfo) {
        videoInfo.enhanced = true;
        videoInfo.enhancedAt = new Date();
      }
    }
  }
  
  // Create global instance
  const detectorInstance = new VideoDetector();

  try {
    root.AutoHDProDetector = detectorInstance;
  } catch (_) {}

  if (typeof window !== 'undefined') {
    window.AutoHDProDetector = detectorInstance;
  }

  if (typeof self !== 'undefined') {
    self.AutoHDProDetector = detectorInstance;
  }

  // Export for modules
  if (typeof module !== 'undefined' && module.exports) {
    module.exports = VideoDetector;
  }

})();
