(function initAutoHDProCaptions() {
  const root = typeof globalThis !== 'undefined' ? globalThis
    : typeof self !== 'undefined' ? self
    : typeof window !== 'undefined' ? window
    : typeof global !== 'undefined' ? global
    : {};

  if (root.AutoHDProCaptions) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = root.AutoHDProCaptions.constructor;
    }
    return;
  }

  /**
   * AutoHDPro Captions Engine
   * Manages caption/subtitle settings across different players
   */
  
  class CaptionsEngine {
    constructor() {
      this.logger = new AutoHDProLogger('Captions');
      this.processedVideos = new WeakSet();
    }
  
    /**
     * Disable captions for a video
     */
    async disableCaptions(video, options = {}) {
      const { driver = null, force = false } = options;
      
      // Skip if already processed (unless forced)
      if (!force && this.processedVideos.has(video)) {
        return true;
      }
      
      this.logger.debug('Disabling captions for video');
      
      // Try driver-specific method first
      if (driver && driver.disableCaptions) {
        try {
          const result = await driver.disableCaptions(video);
          if (result) {
            this.processedVideos.add(video);
            this.logger.info('Captions disabled via driver');
            return result;
          }
        } catch (e) {
          this.logger.error('Driver caption disabling failed:', e);
        }
      }
      
      // Try generic methods
      const result = this.disableCaptionsGeneric(video);
      if (result) {
        this.processedVideos.add(video);
      }
      
      return result;
    }
  
    /**
     * Generic caption disabling
     */
    disableCaptionsGeneric(video) {
      let success = false;
      
      // Method 1: HTML5 TextTracks API
      if (this.disableTextTracks(video)) {
        success = true;
      }
      
      // Method 2: Remove track elements
      if (this.removeTrackElements(video)) {
        success = true;
      }
      
      // Method 3: Player-specific APIs
      if (this.disableViaPlayerAPI(video)) {
        success = true;
      }
      
      // Method 4: DOM manipulation
      if (this.hideCaptionElements()) {
        success = true;
      }
      
      // Method 5: Prevent future captions
      this.preventFutureCaptions(video);
      
      if (success) {
        this.logger.info('Captions disabled using generic methods');
      } else {
        this.logger.warn('Could not disable captions - no compatible method found');
      }
      
      return success;
    }
  
    /**
     * Disable text tracks
     */
    disableTextTracks(video) {
      if (!video.textTracks || video.textTracks.length === 0) {
        return false;
      }
      
      let disabled = 0;
      
      // Disable all text tracks
      for (let i = 0; i < video.textTracks.length; i++) {
        const track = video.textTracks[i];
        
        // Check if it's a subtitle/caption track
        if (track.kind === 'subtitles' || 
            track.kind === 'captions' || 
            track.kind === 'descriptions') {
          
          // Set mode to disabled
          track.mode = 'disabled';
          disabled++;
          
          this.logger.debug(`Disabled track: ${track.label || track.language || track.kind}`);
        }
      }
      
      // Monitor for changes and re-disable
      this.monitorTextTracks(video);
      
      return disabled > 0;
    }
  
    /**
     * Remove track elements from DOM
     */
    removeTrackElements(video) {
      const container = video.parentElement;
      if (!container) return false;
      
      // Find and remove track elements
      const tracks = container.querySelectorAll('track');
      let removed = 0;
      
      tracks.forEach(track => {
        if (track.kind === 'subtitles' || 
            track.kind === 'captions' || 
            !track.kind) {
          track.remove();
          removed++;
        }
      });
      
      // Also remove tracks directly under video
      const videoTracks = video.querySelectorAll('track');
      videoTracks.forEach(track => {
        if (track.kind === 'subtitles' || 
            track.kind === 'captions' || 
            !track.kind) {
          track.remove();
          removed++;
        }
      });
      
      return removed > 0;
    }
  
    /**
     * Disable via player APIs
     */
    disableViaPlayerAPI(video) {
      // Try to find player instance
      const players = [
        video.player,
        video._player,
        video.videojs,
        video.plyr,
        video.jwplayer,
        video.flowplayer,
        video.dash,
        video.hls,
        window.player,
        window.videojs && window.videojs(video)
      ].filter(Boolean);
      
      for (const player of players) {
        try {
          // Try different methods
          const methods = [
            () => player.setTextTrackDisplay && player.setTextTrackDisplay(false),
            () => player.captions && (player.captions.active = false),
            () => player.subtitle && (player.subtitle = -1),
            () => player.setSubtitle && player.setSubtitle(-1),
            () => player.setCaptions && player.setCaptions(false),
            () => player.disableCaptions && player.disableCaptions(),
            () => player.textTracks && player.textTracks().tracks_.forEach(t => t.mode = 'disabled'),
            () => player.captionsOff && player.captionsOff(),
            () => player.cc && (player.cc = false)
          ];
          
          for (const method of methods) {
            try {
              method();
              this.logger.debug('Disabled captions via player API');
              return true;
            } catch (e) {
              // Continue trying
            }
          }
        } catch (e) {
          // Continue with next player
        }
      }
      
      return false;
    }
  
    /**
     * Hide caption elements in DOM
     */
    hideCaptionElements() {
      const selectors = [
        // Generic
        '.captions',
        '.subtitles',
        '.cc-text',
        '.caption-container',
        '.subtitle-container',
        
        // YouTube
        '.ytp-caption-window-container',
        '.caption-window',
        '.ytp-caption-segment',
        
        // Netflix
        '.player-timedtext',
        
        // Video.js
        '.vjs-text-track-display',
        '.vjs-text-track',
        
        // JW Player
        '.jw-captions',
        '.jwcaptions',
        
        // Plyr
        '.plyr__captions',
        
        // Flowplayer
        '.fp-captions',
        
        // Custom players
        '[class*="caption"]',
        '[class*="subtitle"]',
        '[id*="caption"]',
        '[id*="subtitle"]'
      ];
      
      let hidden = 0;
      
      selectors.forEach(selector => {
        const elements = document.querySelectorAll(selector);
        elements.forEach(element => {
          // Hide element
          element.style.display = 'none';
          element.style.visibility = 'hidden';
          element.setAttribute('aria-hidden', 'true');
          hidden++;
        });
      });
      
      // Add CSS to keep them hidden
      if (hidden > 0) {
        this.addHideCaptionsCSS();
      }
      
      return hidden > 0;
    }
  
    /**
     * Add CSS to hide captions
     */
    addHideCaptionsCSS() {
      if (document.getElementById('autohdpro-hide-captions-css')) {
        return;
      }
      
      const css = `
        /* AutoHDPro - Hide Captions */
        .captions,
        .subtitles,
        .cc-text,
        .caption-container,
        .subtitle-container,
        .ytp-caption-window-container,
        .player-timedtext,
        .vjs-text-track-display,
        .jw-captions,
        .plyr__captions,
        .fp-captions,
        [class*="caption"]:not(.caption-button):not(.caption-toggle),
        [class*="subtitle"]:not(.subtitle-button):not(.subtitle-toggle) {
          display: none !important;
          visibility: hidden !important;
        }
        
        /* Disable caption buttons */
        .ytp-subtitles-button,
        .vjs-subs-caps-button,
        button[aria-label*="caption" i],
        button[aria-label*="subtitle" i] {
          opacity: 0.5;
          pointer-events: none;
        }
      `;
      
      const style = document.createElement('style');
      style.id = 'autohdpro-hide-captions-css';
      style.textContent = css;
      document.head.appendChild(style);
    }
  
    /**
     * Monitor text tracks for changes
     */
    monitorTextTracks(video) {
      if (!video.textTracks) return;
      
      // Re-disable if tracks change
      video.textTracks.addEventListener('change', () => {
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track.mode !== 'disabled' && 
              (track.kind === 'subtitles' || track.kind === 'captions')) {
            track.mode = 'disabled';
            this.logger.debug('Re-disabled track that was enabled');
          }
        }
      });
      
      // Also monitor addtrack event
      video.textTracks.addEventListener('addtrack', (event) => {
        const track = event.track;
        if (track.kind === 'subtitles' || track.kind === 'captions') {
          track.mode = 'disabled';
          this.logger.debug('Disabled newly added track');
        }
      });
    }
  
    /**
     * Prevent future captions from being enabled
     */
    preventFutureCaptions(video) {
      // Override track mode setter
      if (video.textTracks) {
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track.kind === 'subtitles' || track.kind === 'captions') {
            try {
              Object.defineProperty(track, 'mode', {
                get() {
                  return 'disabled';
                },
                set(value) {
                  // Ignore attempts to enable
                  return 'disabled';
                }
              });
            } catch (e) {
              // Some tracks might be read-only
            }
          }
        }
      }
      
      // Monitor for attribute changes
      const observer = new MutationObserver((mutations) => {
        mutations.forEach((mutation) => {
          if (mutation.type === 'attributes' && 
              mutation.attributeName === 'src' && 
              mutation.target.tagName === 'TRACK') {
            const track = mutation.target;
            if (track.kind === 'subtitles' || track.kind === 'captions') {
              track.remove();
              this.logger.debug('Removed newly added track element');
            }
          }
        });
      });
      
      observer.observe(video.parentElement || video, {
        attributes: true,
        childList: true,
        subtree: true
      });
      
      // Store observer for cleanup
      video._captionObserver = observer;
    }
  
    /**
     * Enable captions (for toggling back on)
     */
    enableCaptions(video) {
      this.logger.debug('Enabling captions for video');
      
      // Enable text tracks
      if (video.textTracks) {
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track.kind === 'subtitles' || track.kind === 'captions') {
            // Try to enable the default or first track
            if (track.default || i === 0) {
              track.mode = 'showing';
              break;
            }
          }
        }
      }
      
      // Remove from processed set
      this.processedVideos.delete(video);
      
      // Disconnect observer if exists
      if (video._captionObserver) {
        video._captionObserver.disconnect();
        delete video._captionObserver;
      }
      
      return true;
    }
  
    /**
     * Toggle captions
     */
    toggleCaptions(video) {
      // Check current state
      let hasEnabledCaptions = false;
      
      if (video.textTracks) {
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track.mode === 'showing' && 
              (track.kind === 'subtitles' || track.kind === 'captions')) {
            hasEnabledCaptions = true;
            break;
          }
        }
      }
      
      if (hasEnabledCaptions) {
        return this.disableCaptions(video, { force: true });
      } else {
        return this.enableCaptions(video);
      }
    }
  
    /**
     * Get caption state
     */
    getCaptionState(video) {
      const state = {
        available: false,
        enabled: false,
        tracks: []
      };
      
      if (video.textTracks && video.textTracks.length > 0) {
        state.available = true;
        
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track.kind === 'subtitles' || track.kind === 'captions') {
            state.tracks.push({
              id: track.id,
              label: track.label,
              language: track.language,
              kind: track.kind,
              mode: track.mode
            });
            
            if (track.mode === 'showing') {
              state.enabled = true;
            }
          }
        }
      }
      
      return state;
    }
  }
  
  // Create global instance
  const captionsInstance = new CaptionsEngine();

  try {
    root.AutoHDProCaptions = captionsInstance;
  } catch (_) {}

  if (typeof window !== 'undefined') {
    window.AutoHDProCaptions = captionsInstance;
  }

  if (typeof self !== 'undefined') {
    self.AutoHDProCaptions = captionsInstance;
  }

  // Export for modules
  if (typeof module !== 'undefined' && module.exports) {
    module.exports = CaptionsEngine;
  }

})();
