(function initAutoHDProYouTubeDriver() {
  const root = typeof globalThis !== 'undefined' ? globalThis
    : typeof self !== 'undefined' ? self
    : typeof window !== 'undefined' ? window
    : typeof global !== 'undefined' ? global
    : {};

  if (root.YouTubeDriver) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = root.YouTubeDriver;
    }
    return;
  }

  // YouTube Driver
  const YT_IGNORED_QUALITIES = new Set(['auto', 'default']);
  
  const YT_QUALITY_HEIGHTS = {
    highres: 4320,
    hd2880: 2880,
    hd2160: 2160,
    hd1440: 1440,
    hd1080: 1080,
    hd960: 960,
    hd720: 720,
    large: 480,
    medium: 360,
    small: 240,
    tiny: 144
  };
  
  const PREFERRED_HEIGHTS = {
    highest: Number.POSITIVE_INFINITY,
    '4320p': 4320,
    '2880p': 2880,
    '2160p': 2160,
    '1440p': 1440,
    '1080p': 1080,
    '960p': 960,
    '720p': 720,
    '480p': 480,
    '360p': 360,
    '240p': 240,
    '144p': 144
  };
  
  const MAX_LEVEL_ATTEMPTS = 15;
  const APPLY_ATTEMPTS = 8;
  const COMMAND_TIMEOUT_MS = 2000;
  
  function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
  
  function sortByHeight(levels) {
    return levels.slice().sort((a, b) => {
      const heightA = YT_QUALITY_HEIGHTS[a] || 0;
      const heightB = YT_QUALITY_HEIGHTS[b] || 0;
      return heightB - heightA;
    });
  }
  
  function sendYouTubeCommand(action, payload = {}, timeout = COMMAND_TIMEOUT_MS) {
    return new Promise((resolve) => {
      const responseEvent = `AutoHDPro:YouTubeResponse:${Date.now()}:${Math.random().toString(16).slice(2)}`;
  
      const cleanup = () => window.removeEventListener(responseEvent, handler);
      const handler = (event) => {
        clearTimeout(timer);
        cleanup();
        resolve(event.detail || {});
      };
  
      window.addEventListener(responseEvent, handler, { once: true });
  
      const timer = setTimeout(() => {
        cleanup();
        resolve({ success: false, error: 'timeout' });
      }, timeout);
  
      window.dispatchEvent(new CustomEvent('AutoHDPro:YouTubeCommand', {
        detail: {
          action,
          payload,
          responseEvent
        }
      }));
    });
  }
  
  async function getQualityLevels() {
    for (let attempt = 0; attempt < MAX_LEVEL_ATTEMPTS; attempt++) {
      const response = await sendYouTubeCommand('getAvailableQualityLevels');
      const levels = Array.isArray(response?.levels) ? response.levels : [];
      if (response?.success && levels.length) {
        return Array.from(new Set(sortByHeight(levels)));
      }
      await delay(200);
    }
    return [];
  }
  
  function selectQualityLevel(levels, preferredQuality) {
    if (!levels.length) {
      return null;
    }
  
    const normalizedPreference = String(preferredQuality || 'highest').toLowerCase();
    const sorted = sortByHeight(levels);
  
    const preferHeight = PREFERRED_HEIGHTS[normalizedPreference];
    if (!preferHeight || !Number.isFinite(preferHeight)) {
      return sorted[0];
    }
  
    const aboveOrEqual = sorted
      .filter(level => (YT_QUALITY_HEIGHTS[level] || 0) >= preferHeight)
      .sort((a, b) => (YT_QUALITY_HEIGHTS[a] || 0) - (YT_QUALITY_HEIGHTS[b] || 0));
  
    if (aboveOrEqual.length > 0) {
      return aboveOrEqual[0];
    }
  
    return sorted[0];
  }
  
  async function applyQualityWithVerification(levels, quality, log) {
    if (!quality) {
      return false;
    }
  
    const fallbacks = (levels || []).filter(level => level && level !== quality);
    const totalAttempts = (fallbacks.length + 1) * APPLY_ATTEMPTS;
    const timeout = Math.max(COMMAND_TIMEOUT_MS, totalAttempts * 250 + 1000);
  
    const response = await sendYouTubeCommand('setPlaybackQuality', {
      quality,
      fallbacks,
      attempts: APPLY_ATTEMPTS,
      delayMs: 200
    }, timeout);
  
    if (response?.success) {
      if (response.appliedQuality && response.appliedQuality !== quality) {
        log('info', 'Applied fallback YouTube quality', {
          requested: quality,
          applied: response.appliedQuality
        });
      }
      return true;
    }
  
    if (response?.error) {
      log('warn', 'Page quality application failed', response.error);
    }
  
    if (response?.currentQuality) {
      log('info', 'Current YouTube quality after attempt', {
        currentQuality: response.currentQuality,
        currentHeight: response.currentHeight || YT_QUALITY_HEIGHTS[response.currentQuality] || 0
      });
    }
  
    return false;
  }
  
  const driver = {
    matches(location) {
      return location.hostname.includes('youtube.com') || location.hostname.includes('youtu.be');
    },

    async setMaxQuality(video, options) {
      const log = (level, ...args) => {
        const logger = typeof AutoHDProLogger === 'function'
          ? (window.__AutoHDProYoutubeLogger || (window.__AutoHDProYoutubeLogger = new AutoHDProLogger('YouTubeDriver')))
          : null;
  
        if (logger && typeof logger[level] === 'function') {
          logger[level](...args);
        } else if (level === 'error') {
          console.error('AutoHDPro: [YouTube]', ...args);
        } else if (level === 'warn') {
          console.warn('AutoHDPro: [YouTube]', ...args);
        } else {
          console.info('AutoHDPro: [YouTube]', ...args);
        }
      };
  
      const preferredQuality = options?.preferredQuality || 'highest';
      const levels = await getQualityLevels();
  
      if (!levels.length) {
        log('warn', 'No quality levels available');
        return false;
      }
  
      const selected = selectQualityLevel(levels, preferredQuality);
      const result = await applyQualityWithVerification(levels, selected, log);
  
      if (!result && levels[0] && levels[0] !== selected) {
        log('warn', 'Preferred quality failed, falling back to highest', { selected, fallback: levels[0] });
        return applyQualityWithVerification(levels, levels[0], log);
      }
  
      if (result) {
        log('info', 'Applied YouTube quality', { selected, preferredQuality, available: levels });
      }
  
      return result;
    },
  
    async disableCaptions(video) {
      const player = document.querySelector('#movie_player');
      if (player && player.unloadModule) {
        try {
          player.unloadModule('captions');
          player.unloadModule('cc');
        } catch (_) {}
      }
  
      const ccButton = document.querySelector('.ytp-subtitles-button');
      if (ccButton && ccButton.getAttribute('aria-pressed') === 'true') {
        ccButton.click();
      }
      return true;
    }
  };

  root.YouTubeDriver = driver;

  if (typeof window !== 'undefined') {
    window.YouTubeDriver = driver;
  }

  if (typeof self !== 'undefined') {
    self.YouTubeDriver = driver;
  }

  if (typeof module !== 'undefined' && module.exports) {
    module.exports = driver;
  }
})();
