(function initAutoHDProXvideosDriver() {
  const root = typeof globalThis !== 'undefined' ? globalThis
    : typeof self !== 'undefined' ? self
    : typeof window !== 'undefined' ? window
    : typeof global !== 'undefined' ? global
    : {};

  if (root.XvideosDriver) {
    if (typeof module !== 'undefined' && module.exports) {
      module.exports = root.XvideosDriver;
    }
    return;
  }

  // XVideos Driver
  (function() {
    'use strict';
  
    const QUALITY_PRIORITY = {
      highest: Number.POSITIVE_INFINITY,
      '4320p': 4320,
      '2880p': 2880,
      '2160p': 2160,
      '1440p': 1440,
      '1080p': 1080,
      '960p': 960,
      '720p': 720,
      '480p': 480,
      '360p': 360,
      '240p': 240,
      '144p': 144
    };

    const QUALITY_OPTION_SELECTOR = 'li[id^="level"]';
    const SETTINGS_BUTTON_SELECTOR = '.settings-btn, [data-role="settings"], [data-qa="player-settings-button"]';
    const QUALITY_MENU_TRIGGER_SELECTOR = '#quality, [data-role="quality"]';
    const HOST_PATTERNS = ['xvideos', 'xnxx'];
  
    let loggerInstance = null;
  
    function getLogger() {
      if (!loggerInstance && typeof AutoHDProLogger === 'function') {
        loggerInstance = new AutoHDProLogger('XvideosDriver');
      }
      return loggerInstance;
    }
  
    function delay(ms) {
      return new Promise(resolve => setTimeout(resolve, ms));
    }
  
    function waitForValue(getter, timeout = 4000, interval = 100) {
      return new Promise(resolve => {
        const deadline = Date.now() + timeout;
  
        function check() {
          let value = null;
          try {
            value = getter();
          } catch (_) {}
  
          if (value) {
            resolve(value);
            return;
          }
  
          if (Date.now() >= deadline) {
            resolve(null);
            return;
          }
  
          setTimeout(check, interval);
        }
  
        check();
      });
    }
  
    function normalizeQualityLabel(label) {
      if (label == null) {
        return null;
      }
      const normalized = String(label).trim().toLowerCase();
      if (!normalized || normalized === 'auto') {
        return null;
      }
      const match = normalized.match(/(\d{3,4})/);
      if (!match) {
        return null;
      }
      return match[1] + 'p';
    }
  
    function extractHeight(value) {
      if (value == null) {
        return 0;
      }
      if (typeof value === 'number' && Number.isFinite(value)) {
        return value;
      }
      const match = String(value).match(/(\d{3,4})/);
      return match ? parseInt(match[1], 10) : 0;
    }
  
    function isElementVisible(element) {
      if (!element) {
        return false;
      }
      if (element.offsetParent) {
        return true;
      }
      const style = window.getComputedStyle(element);
      return style && style.display !== 'none' && style.visibility !== 'hidden' && style.opacity !== '0';
    }
  
    function getPlayerRoot(video) {
      if (!video || typeof video.closest !== 'function') {
        return document;
      }
      return video.closest('#hlsplayer, #html5video, .embed-responsive, .player-container, .player') || document;
    }
  
    function getSettingsButton(root) {
      return root.querySelector(SETTINGS_BUTTON_SELECTOR) || document.querySelector(SETTINGS_BUTTON_SELECTOR);
    }
  
    function getQualityMenuTrigger(root) {
      return root.querySelector(QUALITY_MENU_TRIGGER_SELECTOR) || document.querySelector(QUALITY_MENU_TRIGGER_SELECTOR);
    }
  
    function getQualityMenu(root) {
      const menus = root.querySelectorAll('ul.settings-menu.secondary-menu');
      for (let i = 0; i < menus.length; i++) {
        const menu = menus[i];
        if (menu.querySelector(QUALITY_OPTION_SELECTOR)) {
          return menu;
        }
      }
      return null;
    }
  
    function collectQualityOptions(root) {
      const menu = getQualityMenu(root);
      if (!menu) {
        return [];
      }
  
      const nodes = menu.querySelectorAll(QUALITY_OPTION_SELECTOR);
      const options = [];
  
      for (let i = 0; i < nodes.length; i++) {
        const node = nodes[i];
        const labelNode = node.querySelector('.label-text') || node.querySelector('.val') || node;
        const rawLabel = labelNode ? labelNode.textContent : '';
        const dataLabel = node.getAttribute('data-quality') || node.getAttribute('data-quality-option') || rawLabel;
        const label = normalizeQualityLabel(dataLabel);
        const height = extractHeight(dataLabel);
  
        if (!height) {
          continue;
        }
  
        const option = {
          label: label || (height + 'p'),
          height,
          element: node,
          active: node.classList.contains('active')
        };
  
        if (options.some(candidate => candidate.height === option.height)) {
          continue;
        }
  
        options.push(option);
      }
  
      return options;
    }
  
    function selectPreferredOption(options, preference) {
      if (!options.length) {
        return null;
      }
  
      const normalizedPreference = String(preference || 'highest').toLowerCase();
      const targetHeight = QUALITY_PRIORITY.hasOwnProperty(normalizedPreference)
        ? QUALITY_PRIORITY[normalizedPreference]
        : Number.POSITIVE_INFINITY;
  
      const sorted = options.slice().sort((a, b) => b.height - a.height);
  
      if (!Number.isFinite(targetHeight)) {
        return sorted[0];
      }
  
      const candidate = sorted.find(option => option.height >= targetHeight);
      return candidate || sorted[0];
    }
  
    async function openQualityMenu(root, logger) {
      const button = getSettingsButton(root);
      const trigger = getQualityMenuTrigger(root);
  
      if (!button || !trigger) {
        logger?.warn('Xvideos controls missing', { hasButton: Boolean(button), hasTrigger: Boolean(trigger) });
        return null;
      }
  
      const mainMenuGetter = () => root.querySelector('ul.settings-menu:not(.secondary-menu)');
      let mainMenu = mainMenuGetter();
  
      if (!mainMenu || !isElementVisible(mainMenu)) {
        button.dispatchEvent(new MouseEvent('click', { bubbles: true }));
        await delay(150);
        mainMenu = await waitForValue(() => {
          const menu = mainMenuGetter();
          return menu && isElementVisible(menu) ? menu : null;
        });
      }
  
      trigger.dispatchEvent(new MouseEvent('mouseenter', { bubbles: true }));
      trigger.dispatchEvent(new MouseEvent('click', { bubbles: true }));
      await delay(150);
  
      const qualityMenu = await waitForValue(() => {
        const menu = getQualityMenu(root);
        return menu && isElementVisible(menu) ? menu : null;
      });
  
      return qualityMenu || getQualityMenu(root);
    }
  
    async function applyQualityOption(option, video, logger) {
      if (!option) {
        return false;
      }
  
      const root = getPlayerRoot(video);
  
      const hydrateOption = () => {
        const options = collectQualityOptions(root);
        const match = options.find(candidate => {
          if (candidate.height && option.height && candidate.height === option.height) {
            return true;
          }
          if (candidate.label && option.label && candidate.label === option.label) {
            return true;
          }
          return false;
        });
        return match || option;
      };
  
      option = hydrateOption();
  
      if (option.active) {
        return true;
      }
  
      const menu = await openQualityMenu(root, logger);
      if (!menu) {
        logger?.warn('Xvideos quality menu not available');
        return false;
      }
  
      option = hydrateOption();
  
      if (!option || !option.element) {
        logger?.warn('Xvideos quality option element missing after menu open');
        return false;
      }
  
      try {
        option.element.dispatchEvent(new MouseEvent('mouseenter', { bubbles: true }));
        option.element.dispatchEvent(new MouseEvent('click', { bubbles: true }));
        await delay(250);
  
        if (option.element.classList.contains('active')) {
          return true;
        }
  
        option.element.click();
        await delay(200);
  
        if (option.element.classList.contains('active')) {
          return true;
        }
  
        const refreshed = hydrateOption();
        if (refreshed && refreshed.element && refreshed.element.classList.contains('active')) {
          return true;
        }
  
        return false;
      } catch (error) {
        logger?.warn('Xvideos option interaction failed', { error: error?.message || error });
        return false;
      }
    }
  
    async function setMaxQualityInternal(video, options) {
      const logger = getLogger();
      const preferredQuality = options && options.preferredQuality ? options.preferredQuality : 'highest';
      const root = getPlayerRoot(video);
  
      await waitForValue(() => getQualityMenu(root), 5000, 150);
  
      const qualityOptions = collectQualityOptions(root);
      if (!qualityOptions.length) {
        logger?.warn('No XVideos quality options detected');
        return false;
      }
  
      const target = selectPreferredOption(qualityOptions, preferredQuality);
      if (!target) {
        logger?.warn('Unable to determine XVideos target quality', { preferredQuality });
        return false;
      }
  
      logger?.info('Attempting XVideos quality upgrade', {
        preferredQuality,
        available: qualityOptions.map(option => option.label),
        target: target.label
      });
  
      const applied = await applyQualityOption(target, video, logger);
      if (applied) {
        logger?.info('XVideos quality applied', { label: target.label, height: target.height });
      } else {
        logger?.warn('XVideos quality application failed', { label: target.label, height: target.height });
      }
  
      return applied;
    }
  
    async function disableCaptionsInternal(video) {
      if (video && video.textTracks) {
        let disabled = false;
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track && track.mode && track.mode.toLowerCase() !== 'disabled') {
            try {
              track.mode = 'disabled';
              disabled = true;
            } catch (_) {}
          }
        }
        if (disabled) {
          return true;
        }
      }
  
      const container = getPlayerRoot(video);
      const selectors = [
        '[data-action="toggle-captions"]',
        '[data-type="captions"]',
        '.js-cc-toggle',
        '.vjs-subs-caps-button'
      ];
  
      for (let i = 0; i < selectors.length; i++) {
        const element = container.querySelector(selectors[i]);
        if (!element) {
          continue;
        }
        const pressed = element.getAttribute('aria-pressed') || element.getAttribute('aria-checked');
        if (pressed === 'false') {
          return true;
        }
        try {
          element.dispatchEvent(new MouseEvent('click', { bubbles: true }));
          return true;
        } catch (_) {}
      }
  
      return false;
    }
  
    const driver = {
      matches(location) {
        const hostname = (location && location.hostname) ? location.hostname.toLowerCase() : '';
        if (!hostname) {
          return false;
        }
        return HOST_PATTERNS.some(pattern => hostname.includes(pattern));
      },

      async setMaxQuality(video, options) {
        const applied = await setMaxQualityInternal(video, options);

        if (!applied) {
          return false;
        }

        video.addEventListener('playing', () => {
          const logger = getLogger();
          logger?.info('XVideos video started playing, scheduling quality re-check');
          chrome.runtime.sendMessage({ type: 'RE_CHECK_PAGE' });
        }, { once: true });
  
        return true;
      },
  
      async disableCaptions(video) {
        return disableCaptionsInternal(video);
      }
    };

    root.XvideosDriver = driver;

    if (typeof window !== 'undefined') {
      window.XvideosDriver = driver;
    }

    if (typeof self !== 'undefined') {
      self.XvideosDriver = driver;
    }

    if (typeof module !== 'undefined' && module.exports) {
      module.exports = driver;
    }

  })();

})();
