// Xhamster Driver
(function() {
  'use strict';

  const QUALITY_PRIORITY = {
    highest: Number.POSITIVE_INFINITY,
    '4320p': 4320,
    '2880p': 2880,
    '2160p': 2160,
    '1440p': 1440,
    '1080p': 1080,
    '960p': 960,
    '720p': 720,
    '480p': 480,
    '360p': 360,
    '240p': 240,
    '144p': 144
  };

  const KEYS_TO_EXPLORE = ['mediaDefinitions', 'mediaDefinition', 'sources', 'source', 'qualities', 'levels', 'items', 'formats', 'files', 'streams', 'variants'];
  const CONTROLLER_METHODS = ['setQuality', 'changeQuality', 'selectQuality', 'chooseQuality', 'setPlaybackQuality', 'updateQuality'];
  const DOM_SELECTORS = [
    '[data-qa="video-quality-item"]',
    '.player-quality-option',
    '[data-testid="xhamster-quality-menu"] [data-quality]',
    '[data-quality-option]',
    '[data-quality]'
  ];

  const PLAYER_CONTAINER_SELECTORS = [
    '.player-container-xhamster',
    '.xh-player-container',
    '.xh-player',
    '.player-container',
    '.video-container',
    '#player',
    '#xh-player'
  ];

  let loggerInstance = null;

  function getLogger() {
    if (!loggerInstance && typeof AutoHDProLogger === 'function') {
      loggerInstance = new AutoHDProLogger('XhamsterDriver');
    }
    return loggerInstance;
  }

  function normalizeQualityLabel(label) {
    if (label == null) {
      return null;
    }
    const normalized = String(label).trim().toLowerCase();
    if (!normalized || normalized === 'auto') {
      return null;
    }
    const match = normalized.match(/(\d{3,4})/);
    if (!match) {
      return null;
    }
    return match[1] + 'p';
  }

  function extractHeight(value) {
    if (value == null) {
      return 0;
    }
    if (typeof value === 'number' && Number.isFinite(value)) {
      return value;
    }
    const match = String(value).match(/(\d{3,4})/);
    return match ? parseInt(match[1], 10) : 0;
  }

  function parseDefinition(definition) {
    if (!definition || typeof definition !== 'object') {
      return null;
    }

    const url = definition.url || definition.src || definition.videoUrl || definition.file || definition.playlist || (definition.embed && definition.embed.src);
    const qualityHint = definition.quality || definition.label || definition.text || definition.id || definition.name || definition.format || definition.resolution || definition.height;
    const height = extractHeight(qualityHint);
    if (!url || !height) {
      return null;
    }

    const label = normalizeQualityLabel(qualityHint) || (height + 'p');
    if (!label) {
      return null;
    }

    return { url, label, height };
  }

  function extractDefinitions(input, definitions, visited) {
    if (!input) {
      return;
    }

    if (typeof input === 'string') {
      const trimmed = input.trim();
      if (trimmed.startsWith('{') || trimmed.startsWith('[')) {
        try {
          extractDefinitions(JSON.parse(trimmed), definitions, visited);
        } catch (_) {}
      }
      return;
    }

    if (typeof input !== 'object') {
      return;
    }

    if (visited.has(input)) {
      return;
    }
    visited.add(input);

    if (Array.isArray(input)) {
      input.forEach(item => extractDefinitions(item, definitions, visited));
      return;
    }

    const parsed = parseDefinition(input);
    if (parsed) {
      const exists = definitions.some(def => def.url === parsed.url || (def.label === parsed.label && def.height === parsed.height));
      if (!exists) {
        definitions.push(parsed);
      }
    }

    for (let i = 0; i < KEYS_TO_EXPLORE.length; i++) {
      const key = KEYS_TO_EXPLORE[i];
      if (Object.prototype.hasOwnProperty.call(input, key)) {
        extractDefinitions(input[key], definitions, visited);
      }
    }
  }

  function collectMediaDefinitions(video) {
    const definitions = [];
    const visited = new WeakSet();

    const candidates = [
      window.initials,
      window.initials && window.initials.videoModel,
      window.initialState,
      window.initialData,
      window.__INITIAL_STATE__,
      window.playerSettings,
      window.playerConfig,
      window.xplayer,
      window.xplayer && window.xplayer.config,
      window.xplayer && window.xplayer.sources,
      window.xplayer && window.xplayer.options
    ];

    candidates.forEach(candidate => extractDefinitions(candidate, definitions, visited));

    if (video) {
      const dataSources = video.getAttribute('data-sources') || video.dataset.sources || video.dataset.videoSources;
      if (dataSources) {
        try {
          extractDefinitions(JSON.parse(dataSources), definitions, visited);
        } catch (_) {}
      }

      const container = video.closest('[data-sources]');
      if (container) {
        const containerData = container.getAttribute('data-sources');
        if (containerData) {
          try {
            extractDefinitions(JSON.parse(containerData), definitions, visited);
          } catch (_) {}
        }
      }
    }

    const datasetNodes = document.querySelectorAll('[data-player-sources], [data-quality-sources]');
    datasetNodes.forEach(node => {
      const attr = node.getAttribute('data-player-sources') || node.getAttribute('data-quality-sources');
      if (!attr) {
        return;
      }
      try {
        extractDefinitions(JSON.parse(attr), definitions, visited);
      } catch (_) {}
    });

    const jsonScripts = document.querySelectorAll('script[type="application/json"][data-player-config]');
    jsonScripts.forEach(script => {
      try {
        extractDefinitions(JSON.parse(script.textContent || '[]'), definitions, visited);
      } catch (_) {}
    });

    const videoContainer = document.querySelector('.video-container');
    if (videoContainer && videoContainer.dataset.playerSources) {
      try {
        const sources = JSON.parse(videoContainer.dataset.playerSources);
        extractDefinitions(sources, definitions, visited);
      } catch (_) {}
    }


    return definitions;
  }

  function selectPreferredDefinition(definitions, preference) {
    if (!definitions.length) {
      return null;
    }
    const normalizedPreference = String(preference || 'highest').toLowerCase();
    const targetHeight = QUALITY_PRIORITY.hasOwnProperty(normalizedPreference)
      ? QUALITY_PRIORITY[normalizedPreference]
      : Number.POSITIVE_INFINITY;

    const sorted = definitions.slice().sort((a, b) => b.height - a.height);
    if (!Number.isFinite(targetHeight)) {
      return sorted[0];
    }

    const candidate = sorted.find(def => def.height >= targetHeight);
    return candidate || sorted[0];
  }

  function buildLabelVariants(label) {
    const variants = new Set();
    const normalized = normalizeQualityLabel(label);
    if (!normalized) {
      return variants;
    }
    const numeric = normalized.replace(/[^0-9]/g, '');
    variants.add(normalized);
    variants.add(normalized.toUpperCase());
    if (numeric) {
      variants.add(numeric);
      variants.add(numeric + 'p');
    }
    return variants;
  }

  function getControllerCandidates() {
    const candidates = [];
    const push = (candidate) => {
      if (!candidate) {
        return;
      }
      if (Array.isArray(candidate)) {
        candidate.forEach(push);
        return;
      }
      candidates.push(candidate);
    };

    push(window.xplayer);
    push(window.xplayer && window.xplayer.api);
    push(window.xplayer && window.xplayer.controller);
    push(window.xplayer && window.xplayer.qualityController);
    push(window.xplayer && window.xplayer.players && window.xplayer.players.main);
    push(window.playerSettings);
    push(window.playerSettings && window.playerSettings.controller);
    push(window.playerSettings && window.playerSettings.api);
    push(window.playerConfig);

    return candidates.filter(Boolean);
  }

  function applyViaControllers(label) {
    const variants = buildLabelVariants(label);
    if (!variants.size) {
      return false;
    }

    const controllers = getControllerCandidates();
    let applied = false;

    controllers.forEach(controller => {
      if (applied || !controller) {
        return;
      }
      for (let i = 0; i < CONTROLLER_METHODS.length; i++) {
        const method = CONTROLLER_METHODS[i];
        if (typeof controller[method] !== 'function') {
          continue;
        }
        for (const variant of variants) {
          try {
            const result = controller[method](variant);
            if (result !== false) {
              applied = true;
              return;
            }
          } catch (_) {}
        }
      }
    });

    return applied;
  }

  function applyViaDom(label) {
    const variants = buildLabelVariants(label);
    if (!variants.size) {
      return false;
    }

    for (let i = 0; i < DOM_SELECTORS.length; i++) {
      const elements = document.querySelectorAll(DOM_SELECTORS[i]);
      for (let j = 0; j < elements.length; j++) {
        const element = elements[j];
        const data = element.getAttribute('data-quality') || element.getAttribute('data-quality-option') || element.textContent;
        const normalized = normalizeQualityLabel(data);
        if (normalized && variants.has(normalized)) {
          try {
            element.dispatchEvent(new MouseEvent('click', { bubbles: true }));
            return true;
          } catch (_) {}
        }
      }
    }

    return false;
  }

  function setVideoSource(video, definition) {
    if (!video || !definition || !definition.url) {
      return false;
    }

    try {
      const currentSrc = video.currentSrc || video.src || '';
      if (currentSrc === definition.url) {
        return false;
      }

      const currentTime = Number.isFinite(video.currentTime) ? video.currentTime : 0;
      const wasPaused = video.paused;

      const handleLoaded = () => {
        video.removeEventListener('loadedmetadata', handleLoaded);
        try {
          if (Number.isFinite(currentTime)) {
            video.currentTime = currentTime;
          }
          if (!wasPaused) {
            const playResult = video.play();
            if (playResult && typeof playResult.then === 'function') {
              playResult.catch(() => {});
            }
          }
        } catch (_) {}
      };

      video.addEventListener('loadedmetadata', handleLoaded, { once: true });
      video.src = definition.url;
      return true;
    } catch (_) {
      return false;
    }
  }

  function applyQuality(video, definition, logger) {
    if (!definition) {
      return false;
    }

    const label = definition.label;

    if (applyViaControllers(label)) {
      logger?.debug('Applied Xhamster quality via controller API', { label });
      return true;
    }

    if (applyViaDom(label)) {
      logger?.debug('Applied Xhamster quality via DOM interaction', { label });
      return true;
    }

    if (setVideoSource(video, definition)) {
      logger?.debug('Applied Xhamster quality via source swap', { label });
      return true;
    }

    if (window.AutoHDProQuality && typeof window.AutoHDProQuality.forceQualityBySource === 'function') {
      return window.AutoHDProQuality.forceQualityBySource(video, label);
    }

    return false;
  }

  function resolvePlayerContainer(video) {
    if (!video) {
      return null;
    }

    const selectorList = PLAYER_CONTAINER_SELECTORS.join(', ');

    if (typeof video.closest === 'function') {
      const directMatch = video.closest(selectorList);
      if (directMatch) {
        return directMatch;
      }
    }

    for (let i = 0; i < PLAYER_CONTAINER_SELECTORS.length; i++) {
      const selector = PLAYER_CONTAINER_SELECTORS[i];
      const candidate = document.querySelector(selector);
      if (candidate && candidate.contains(video)) {
        return candidate;
      }
    }

    return video.parentElement || null;
  }

  function waitForPlayerAndApply(video, options) {
    const logger = getLogger();
    const playerContainer = resolvePlayerContainer(video);
    if (!playerContainer) {
      logger.warn('Xhamster player container not found', {
        selectors: PLAYER_CONTAINER_SELECTORS
      });
      return;
    }

    const apply = () => {
      // Remove listener to prevent multiple executions
      playerContainer.removeEventListener('player:ready', apply);
      logger.info('Xhamster player is ready, attempting to set quality.');
      
      // Open settings menu
      const settingsButton = playerContainer.querySelector('[data-qa="player-settings-button"]');
      if (settingsButton) {
        settingsButton.click();
        
        // Wait for menu to render
        setTimeout(() => {
          setMaxQuality(video, options);
        }, 250);
      } else {
        logger.warn('Could not find Xhamster settings button.');
      }
    };

    playerContainer.addEventListener('player:ready', apply, { once: true });
  }

  // Main driver logic
  async function setMaxQuality(video, options) {
    const logger = getLogger();
    const preferredQuality = options && options.preferredQuality ? options.preferredQuality : 'highest';

    const definitions = collectMediaDefinitions(video);
    if (!definitions.length) {
      logger?.warn('No Xhamster media definitions found');
      return false;
    }

    const target = selectPreferredDefinition(definitions, preferredQuality);
    if (!target) {
      logger?.warn('Unable to determine target quality', { preferredQuality });
      return false;
    }

    logger?.info('Attempting Xhamster quality upgrade', {
      preferredQuality,
      available: definitions.map(def => def.label),
      target
    });

    const result = applyQuality(video, target, logger);
    if (result) {
      logger?.info('Xhamster quality applied', { label: target.label, height: target.height });
    } else {
      logger?.warn('Xhamster quality application failed', { label: target.label, height: target.height });
    }

    return result;
  }

  window.XhamsterDriver = {
    matches(location) {
      const host = location.hostname;
      return host.includes('xhamster.com') || 
             host.includes('xhamster.one') || 
             host.includes('xhamster.desi') ||
             host.includes('xhamster.pro');
    },

    async setMaxQuality(video, options) {
      waitForPlayerAndApply(video, options);

      // Fallback: if video starts playing, re-check.
      video.addEventListener('playing', () => {
        const logger = getLogger();
        logger.info('Xhamster video started playing, re-checking quality.');
        chrome.runtime.sendMessage({ type: 'RE_CHECK_PAGE' });
      }, { once: true });

      // Return true to indicate we've handled it, even though it's async.
      // The core engine doesn't need to retry.
      return true;
    },

    async disableCaptions(video) {
      if (video && video.textTracks) {
        let disabled = false;
        for (let i = 0; i < video.textTracks.length; i++) {
          const track = video.textTracks[i];
          if (track && track.mode && track.mode.toLowerCase() !== 'disabled') {
            try {
              track.mode = 'disabled';
              disabled = true;
            } catch (_) {}
          }
        }
        if (disabled) {
          return true;
        }
      }

      const selectors = [
        '[data-action="toggle-captions"]',
        '[data-type="captions"]',
        '.js-cc-toggle',
        '.vjs-subs-caps-button'
      ];

      for (let i = 0; i < selectors.length; i++) {
        const element = document.querySelector(selectors[i]);
        if (!element) {
          continue;
        }
        const pressed = element.getAttribute('aria-pressed') || element.getAttribute('aria-checked');
        if (pressed === 'false') {
          return true;
        }
        try {
          element.dispatchEvent(new MouseEvent('click', { bubbles: true }));
          return true;
        } catch (_) {}
      }

      return false;
    }
  };
})();
