/**
 * AutoHDPro Content Script
 * Main orchestrator for video enhancement
 */

(async function() {
  'use strict';
  
  // Prevent multiple initializations
  if (window.AutoHDProInitialized) {
    return;
  }
  window.AutoHDProInitialized = true;
  
  // Initialize logger
  const logger = new AutoHDProLogger('Content');
  logger.info('AutoHDPro initializing...');

  const processedActivationSessions = new Set();
  const MAX_ENHANCEMENT_RETRIES = 4;
  const ENHANCEMENT_STATUS_EVENTS = {
    started: 'VIDEO_ENHANCEMENT_STARTED',
    success: 'VIDEO_ENHANCEMENT_SUCCESS',
    failed: 'VIDEO_ENHANCEMENT_FAILED'
  };

  const QUALITY_ALIAS_TABLE = [
    { score: 4320, aliases: ['4320p', '8k', '7680x4320'] },
    { score: 2160, aliases: ['2160p', '4k', 'uhd', '3840x2160'] },
    { score: 1440, aliases: ['1440p', '2k', 'qhd', '2560x1440'] },
    { score: 1080, aliases: ['1080p', 'fhd', 'full hd', '1920x1080', 'hd1080'] },
    { score: 720, aliases: ['720p', '1280x720', 'hd720'] },
    { score: 480, aliases: ['480p', 'sd', '854x480'] },
    { score: 360, aliases: ['360p', '640x360'] },
    { score: 240, aliases: ['240p', '426x240'] },
    { score: 144, aliases: ['144p', '256x144'] }
  ];

  function getQualityScore(label) {
    if (!label) {
      return 0;
    }

    const normalized = String(label).trim().toLowerCase();

    if (normalized === 'hd') {
      return 720;
    }

    if (normalized === 'sd') {
      return 480;
    }

    const dimensionMatch = normalized.match(/(\d{3,4})\s*[x×]\s*(\d{3,4})/);
    if (dimensionMatch) {
      return parseInt(dimensionMatch[2], 10);
    }

    const progressiveMatch = normalized.match(/(\d{3,4})\s*p/);
    if (progressiveMatch) {
      return parseInt(progressiveMatch[1], 10);
    }

    for (const entry of QUALITY_ALIAS_TABLE) {
      if (entry.aliases.some(alias => normalized.includes(alias))) {
        return entry.score;
      }
    }

    const genericDigits = normalized.match(/(\d{3,4})/);
    if (genericDigits) {
      return parseInt(genericDigits[1], 10);
    }

    return 0;
  }

  function formatQualityLabel(label) {
    if (!label) {
      return 'unknown';
    }
    const trimmed = String(label).trim();
    return trimmed.length ? trimmed : 'unknown';
  }

  const SUPPORTED_HOST_PATTERNS = [
    'youtube.com',
    'youtu.be',
    'twitch.tv',
    'vimeo.com',
    'pornhub.com',
    'xhamster.com',
    'xvideos',
    'xnxx',
    'redtube.com',
    'youporn.com'
  ];

  // Available drivers
  const drivers = [
    window.YouTubeDriver,
    window.TwitchDriver,
    window.VimeoDriver,
    window.PornhubDriver,
    window.XhamsterDriver,
    window.XvideosDriver
  ].filter(Boolean);
  
  // Get current driver for this site
  const currentDriver = drivers.find(driver => driver.matches(window.location));
  
  if (currentDriver) {
    logger.info(`Using driver: ${currentDriver.constructor?.name || 'Site-specific'} driver`);
  } else {
    logger.info('Using generic driver');
  }
  
  // Initialize storage and get settings
  await AutoHDProStorage.init();
  let settings = await AutoHDProStorage.getSettings();
  const hostname = window.location.hostname;
  let siteOverride = await AutoHDProStorage.getSiteOverride(hostname);
  const shortsBlocker = window.AutoHDProShortsBlocker || null;

  setupProActivationBridge();

  // Performance check - only run on supported sites
  const isSupportedSite = Boolean(currentDriver) ||
    SUPPORTED_HOST_PATTERNS.some(pattern => hostname.includes(pattern));
  
  if (!isSupportedSite) {
    logger.info('Site not supported, skipping detection');
    if (shortsBlocker) {
      try {
        shortsBlocker.disable();
      } catch (error) {
        logger.warn('Failed to disable shorts blocker on unsupported site', error);
      }
    }
    return;
  }

  const isEnhancementAllowed = () => getEnhancementState() === 'enabled';


  function applyShortsPreference(enabled) {
    if (!shortsBlocker) {
      return;
    }

    try {
      if (!shortsBlocker.isSupported(window.location)) {
        shortsBlocker.disable();
        return;
      }

      shortsBlocker.sync(Boolean(enabled));
    } catch (error) {
      logger.warn('Shorts blocker toggle failed', error);
    }
  }

  let detectorActive = false;

  async function notifyEnhancementEvent(eventKey) {
    const messageType = ENHANCEMENT_STATUS_EVENTS[eventKey];
    if (!messageType) {
      return;
    }

    try {
      await chrome.runtime.sendMessage({ type: messageType });
    } catch (error) {
      logger.debug('Failed to notify enhancement status', {
        eventKey,
        error: error?.message || error
      });
    }
  }

  // Process video function
  async function processVideo(videoInfo, skipUsageCheck = false) {
    const video = videoInfo.element;

    if (videoInfo.processing) {
      logger.debug('Video processing already in progress');
      return;
    }

    if (videoInfo.enhanced) {
      logger.debug('Video already enhanced');
      return;
    }

    if (!isEnhancementAllowed()) {
      logger.debug('Skipping video processing because enhancements are disabled');
      return;
    }

    videoInfo.processing = true;

    try {
      logger.info('Processing video:', videoInfo.id);
      await notifyEnhancementEvent('started');

      // Only check usage for new videos, not when settings change
      if (!skipUsageCheck && !videoInfo.usageChecked) {
        // Check usage before HD conversion
        const usageResponse = await chrome.runtime.sendMessage({
          type: 'HD_CONVERSION_ATTEMPT'
        });

        if (!usageResponse.success) {
          // User has exceeded limit, show upgrade prompt
          logger.info('HD conversion limit reached, showing upgrade prompt');
          showUpgradePrompt(usageResponse.usage, usageResponse.message);
          return;
        }

        // Mark that we've checked usage for this video
        videoInfo.usageChecked = true;

        // Show usage notification if getting close to limit
        if (
          usageResponse.usage &&
          usageResponse.usage.plan === 'free' &&
          typeof usageResponse.usage.remaining === 'number' &&
          usageResponse.usage.remaining <= 50
        ) {
          showUsageNotification(usageResponse.usage);
        }
      }

      const qualityApiAvailable = AutoHDProQuality && typeof AutoHDProQuality.getCurrentQuality === 'function';
      const qualityRequested = Boolean(settings.autoQuality);
      const captionsRequested = Boolean(settings.globalCaptionsOff);
      const initialQualityLabel = qualityApiAvailable ? AutoHDProQuality.getCurrentQuality(video) : 'unknown';
      let finalQualityLabel = initialQualityLabel;
      let qualityImproved = false;
      let qualityApplied = false;
      let captionsDisabled = false;

      // Set quality
      if (qualityRequested) {
        try {
          qualityApplied = await AutoHDProQuality.setMaxQuality(video, {
            driver: currentDriver,
            preferredQuality: siteOverride?.quality || settings.preferredQuality
          });

          if (qualityApplied) {
            await new Promise(resolve => setTimeout(resolve, 120));
          }

          if (qualityApiAvailable) {
            finalQualityLabel = AutoHDProQuality.getCurrentQuality(video);
          }

          if (qualityApplied && getQualityScore(finalQualityLabel) > getQualityScore(initialQualityLabel)) {
            qualityImproved = true;
          } else if (qualityApplied) {
            logger.debug('Video already at highest available quality', {
              videoId: videoInfo.id,
              quality: formatQualityLabel(finalQualityLabel)
            });
          } else {
            logger.debug('Quality not yet adjustable, will retry', {
              videoId: videoInfo.id,
              initialQuality: formatQualityLabel(initialQualityLabel)
            });
          }
        } catch (e) {
          logger.error('Failed to set quality:', e);
        }
      }

      // Disable captions
      if (captionsRequested) {
        try {
          captionsDisabled = await AutoHDProCaptions.disableCaptions(video, {
            driver: currentDriver
          });

          if (captionsDisabled) {
            logger.info('Captions disabled for video:', videoInfo.id);
          }
        } catch (e) {
          logger.error('Failed to disable captions:', e);
        }
      }

      const nothingRequested = !qualityRequested && !captionsRequested;
      const qualitySuccessful = qualityRequested ? qualityApplied : false;
      const captionsSuccessful = captionsRequested ? captionsDisabled : false;
      const enhancementComplete = nothingRequested || qualitySuccessful || captionsSuccessful;

      if (!enhancementComplete) {
        const attempts = videoInfo.retryCount || 0;

        if (attempts < MAX_ENHANCEMENT_RETRIES) {
          const nextAttempt = attempts + 1;
          videoInfo.retryCount = nextAttempt;

          if (videoInfo.retryTimer) {
            clearTimeout(videoInfo.retryTimer);
          }

          const retryDelay = Math.min(1500, 250 * Math.pow(2, attempts));
          videoInfo.retryTimer = setTimeout(() => {
            videoInfo.retryTimer = null;
            processVideo(videoInfo, true).catch((error) => {
              logger.error('Retry processing failed', error);
            });
          }, retryDelay);

          logger.debug('Enhancement not ready, scheduling retry', {
            videoId: videoInfo.id,
            attempt: nextAttempt,
            delayMs: retryDelay
          });
        } else {
          logger.warn('Max enhancement retries reached, giving up for this video', videoInfo.id);
          videoInfo.retryCount = 0;
          await notifyEnhancementEvent('failed');
        }

        return;
      }

      if (videoInfo.retryTimer) {
        clearTimeout(videoInfo.retryTimer);
        videoInfo.retryTimer = null;
      }
      videoInfo.retryCount = 0;

      if (qualityRequested) {
        logger.info(
          `Quality summary for ${videoInfo.id}: ${formatQualityLabel(initialQualityLabel)} → ${formatQualityLabel(finalQualityLabel)}`,
          {
            applied: Boolean(qualityApplied),
            improved: qualityImproved
          }
        );
      }

      if (qualitySuccessful) {
        await notifyEnhancementEvent('success');
      }

      if (qualityImproved) {
        const captionStatus = settings.globalCaptionsOff ? `captionsDisabled=${Boolean(captionsDisabled)}` : 'captionsDisabled=not-requested';
        logger.info(
          `Upscaled video ${videoInfo.id} from ${formatQualityLabel(initialQualityLabel)} to ${formatQualityLabel(finalQualityLabel)} (${captionStatus})`
        );
      }

      // Mark as enhanced
      AutoHDProDetector.markEnhanced(video);

      // Update statistics (only for new enhancements, not re-processing)
      if (!skipUsageCheck) {
        try {
          const statsResponse = await chrome.runtime.sendMessage({
            type: 'VIDEO_DETECTED'
          });

          if (statsResponse?.success && typeof statsResponse.videosEnhanced === 'number') {
            logger.info(`Videos enhanced so far: ${statsResponse.videosEnhanced}`);
          }
        } catch (error) {
          logger.warn('Failed to update enhancement statistics', error);
        }
      }
    } finally {
      videoInfo.processing = false;
    }
  }
  
  function getEnhancementState() {
    if (!settings.enabled) {
      return 'disabled-global';
    }

    if (siteOverride && siteOverride.enabled === false) {
      return 'disabled-site';
    }

    return 'enabled';
  }

  function reprocessTrackedVideos() {
    AutoHDProDetector.getVideos().forEach((videoInfo) => {
      videoInfo.enhanced = false;
      processVideo(videoInfo, true);
    });
  }

  let lastEnhancementState = null;

  function syncEnhancementState({ reprocess = false } = {}) {
    const state = getEnhancementState();
    const allowEnhancement = state === 'enabled';

    if (state !== lastEnhancementState) {
      if (state === 'disabled-global') {
        logger.info('AutoHDPro is disabled');
      } else if (state === 'disabled-site') {
        logger.info('AutoHDPro is disabled for this site');
      } else {
        logger.info('AutoHDPro enhancements enabled');
      }
      lastEnhancementState = state;
    }

    if (!allowEnhancement) {
      if (detectorActive) {
        AutoHDProDetector.stop();
        detectorActive = false;
      }
      applyShortsPreference(false);
      return;
    }

    if (!detectorActive) {
      AutoHDProDetector.start();
      detectorActive = true;
    }

    applyShortsPreference(settings.removeShorts);

    if (reprocess) {
      reprocessTrackedVideos();
    }
  }

  // Set up video detection
  AutoHDProDetector.on((event, videoInfo) => {
    switch (event) {
      case 'detected':
        logger.debug('Video detected:', videoInfo.id);
        // Process immediately if ready
        if (videoInfo.element.readyState >= 2) {
          processVideo(videoInfo);
        }
        break;
        
      case 'ready':
      case 'metadata':
        // Process when video is ready
        processVideo(videoInfo);
        break;
        
      case 'srcchange':
        // Re-process on source change
        videoInfo.enhanced = false;
        // Don't increment usage counter for same video element with new source
        processVideo(videoInfo, true);
        break;
    }
  });
  
  // Start detection based on current settings
  syncEnhancementState();
  
  // Listen for messages from popup/background
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    switch (message.type) {
      case 'TOGGLE_ENABLED':
        {
          const previousState = getEnhancementState();
          settings.enabled = message.enabled;
          const currentState = getEnhancementState();
          const shouldReprocess = currentState === 'enabled' && previousState !== 'enabled';
          syncEnhancementState({ reprocess: shouldReprocess });
        }
        sendResponse({ success: true });
        break;
        
      case 'REFRESH_SETTINGS':
        (async () => {
          try {
            const previousSettings = settings ? { ...settings } : null;
            const previousOverride = siteOverride ? { ...siteOverride } : null;
            const previousState = getEnhancementState();
            const newSettings = await AutoHDProStorage.getSettings();
            const newOverride = await AutoHDProStorage.getSiteOverride(hostname);

            settings = newSettings;
            siteOverride = newOverride;

            if (previousSettings) {
              const qualityChanged =
                newSettings.globalCaptionsOff !== previousSettings.globalCaptionsOff ||
                newSettings.autoQuality !== previousSettings.autoQuality ||
                newSettings.preferredQuality !== previousSettings.preferredQuality;

              const overrideChanged =
                (previousOverride?.enabled ?? true) !== (newOverride?.enabled ?? true) ||
                (previousOverride?.captionsOff ?? true) !== (newOverride?.captionsOff ?? true) ||
                (previousOverride?.quality ?? null) !== (newOverride?.quality ?? null);

              const currentState = getEnhancementState();
              const becameActive = currentState === 'enabled' && previousState !== 'enabled';

              syncEnhancementState({
                reprocess: becameActive || qualityChanged || overrideChanged
              });
            } else {
              syncEnhancementState();
            }

            sendResponse({ success: true });
          } catch (error) {
            logger.error('Failed to refresh settings via message', error);
            sendResponse({ success: false, error: error?.message || 'Failed to refresh settings' });
          }
        })();
        return true; // Async response
        
      case 'GET_VIDEO_COUNT':
        sendResponse({
          count: AutoHDProDetector.getVideos().length,
          enhanced: AutoHDProDetector.getVideos().filter(v => v.enhanced).length
        });
        break;

      case 'TOGGLE_CAPTIONS':
        const videos = AutoHDProDetector.getVideos();
        videos.forEach(videoInfo => {
          AutoHDProCaptions.toggleCaptions(videoInfo.element);
        });
        sendResponse({ success: true });
        break;

      case 'SET_REMOVE_SHORTS':
        settings.removeShorts = Boolean(message.enabled);
        syncEnhancementState();
        sendResponse({ success: true });
        break;
    }
  });
  
  // Listen for storage changes
  AutoHDProStorage.onChange((changes) => {
    (async () => {
      const previousState = getEnhancementState();
      let qualityChanged = false;
      let overrideChanged = false;
      let removeShortsChanged = false;

      if ('enabled' in changes) {
        settings.enabled = changes.enabled;
      }

      if ('globalCaptionsOff' in changes) {
        settings.globalCaptionsOff = changes.globalCaptionsOff;
        qualityChanged = true;
      }

      if ('autoQuality' in changes) {
        settings.autoQuality = changes.autoQuality;
        qualityChanged = true;
      }

      if ('preferredQuality' in changes) {
        settings.preferredQuality = changes.preferredQuality;
        qualityChanged = true;
      }

      if ('removeShorts' in changes) {
        settings.removeShorts = changes.removeShorts;
        removeShortsChanged = true;
      }

      if ('subscription' in changes) {
        settings.subscription = changes.subscription;
      }

      if ('siteOverrides' in changes) {
        settings.siteOverrides = changes.siteOverrides;
        const previousOverride = siteOverride ? { ...siteOverride } : null;
        const updatedOverride = await AutoHDProStorage.getSiteOverride(hostname);
        siteOverride = updatedOverride;
        overrideChanged =
          (previousOverride?.enabled ?? true) !== (updatedOverride?.enabled ?? true) ||
          (previousOverride?.captionsOff ?? true) !== (updatedOverride?.captionsOff ?? true) ||
          (previousOverride?.quality ?? null) !== (updatedOverride?.quality ?? null);
      }

      const currentState = getEnhancementState();
      const becameActive = currentState === 'enabled' && previousState !== 'enabled';

      const shouldReprocess =
        (qualityChanged && currentState === 'enabled') ||
        (overrideChanged && currentState === 'enabled') ||
        becameActive;

      const needsSync =
        previousState !== currentState ||
        qualityChanged ||
        overrideChanged ||
        removeShortsChanged;

      if (needsSync) {
        syncEnhancementState({ reprocess: shouldReprocess });
      }
    })();
  });
  
  function setupProActivationBridge() {
    if (window.AutoHDProActivationListener) {
      return;
    }

    window.AutoHDProActivationListener = true;

    window.addEventListener('message', (event) => {
      if (event.origin !== window.location.origin) {
        return;
      }

      const data = event.data;

      if (!data || data.type !== 'AUTOHDPRO_ACTIVATE_PRO' || !data.sessionId) {
        return;
      }

      if (processedActivationSessions.has(data.sessionId)) {
        return;
      }

      processedActivationSessions.add(data.sessionId);
      logger.info('Received AutoHDPro Pro activation signal, validating Stripe session');

      (async () => {
        try {
          const response = await chrome.runtime.sendMessage({
            type: 'ACTIVATE_PRO_SUBSCRIPTION',
            sessionId: data.sessionId,
          });

          if (response?.success) {
            showActivationToast('AutoHDPro Pro is now active! Unlimited HD is unlocked.');
          } else if (response?.error) {
            showActivationToast(response.error, 'error');
          }
        } catch (error) {
          logger.error('Failed to activate subscription:', error);
          showActivationToast('Activation failed. Please refresh or contact support.', 'error');
        }
      })();
    });
  }

  // Show upgrade prompt when limit is reached
  function showUpgradePrompt(usage, serverMessage) {
    const existingPrompt = document.getElementById('autohdpro-upgrade-prompt');
    if (existingPrompt) {
      existingPrompt.remove();
    }

    const prompt = document.createElement('div');
    prompt.id = 'autohdpro-upgrade-prompt';
    prompt.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      width: 320px;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      padding: 20px;
      border-radius: 12px;
      box-shadow: 0 10px 30px rgba(0,0,0,0.3);
      z-index: 999999;
      font-family: system-ui, -apple-system, sans-serif;
      animation: slideIn 0.3s ease-out;
    `;

    const limitText = typeof usage?.limit === 'number' ? usage.limit : 'your free plan';
    const conversionsUsed = typeof usage?.conversions === 'number' ? usage.conversions : null;
    const totalConverted = typeof usage?.totalConversions === 'number' ? usage.totalConversions : null;
    const resetDateText = usage?.resetDate ? new Date(usage.resetDate).toLocaleDateString() : null;
    const message = serverMessage || `You've used all ${limitText} free HD conversions this month.`;

    prompt.innerHTML = `
      <style>
        @keyframes slideIn {
          from { transform: translateX(400px); opacity: 0; }
          to { transform: translateX(0); opacity: 1; }
        }
      </style>
      <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 12px;">
        <h3 style="margin: 0; font-size: 18px; font-weight: 600;">HD Limit Reached!</h3>
        <button id="autohdpro-close-prompt" style="background: none; border: none; color: white; cursor: pointer; font-size: 20px; padding: 0; line-height: 1;">×</button>
      </div>
      <p style="margin: 0 0 16px 0; font-size: 14px; line-height: 1.5; opacity: 0.95;">
        ${message}
      </p>
      <div style="margin-bottom: 16px; font-size: 13px; opacity: 0.9;">
        ${conversionsUsed !== null ? `<div><strong>Monthly conversions:</strong> ${conversionsUsed}${typeof limitText === 'number' ? ` / ${limitText}` : ''}</div>` : ''}
        ${totalConverted !== null ? `<div><strong>Total HD conversions:</strong> ${totalConverted}</div>` : ''}
      </div>
      <div style="background: rgba(255,255,255,0.2); padding: 12px; border-radius: 8px; margin-bottom: 16px;">
        <div style="font-size: 13px; margin-bottom: 6px;">Upgrade to Pro for:</div>
        <ul style="margin: 0; padding-left: 20px; font-size: 13px;">
          <li>Unlimited HD conversions</li>
          <li>Priority support</li>
          <li>Advanced features</li>
        </ul>
      </div>
      <button id="autohdpro-upgrade-btn" style="
        width: 100%;
        background: white;
        color: #667eea;
        border: none;
        padding: 12px;
        border-radius: 8px;
        font-size: 15px;
        font-weight: 600;
        cursor: pointer;
        transition: transform 0.2s;
      ">Upgrade to Pro - $9.99/mo</button>
      ${resetDateText ? `<div style="text-align: center; margin-top: 10px; font-size: 12px; opacity: 0.8;">Resets on: ${resetDateText}</div>` : ''}
    `;

    document.body.appendChild(prompt);

    // Add event listeners
    document.getElementById('autohdpro-close-prompt').addEventListener('click', () => {
      prompt.remove();
    });

    document.getElementById('autohdpro-upgrade-btn').addEventListener('click', () => {
      chrome.runtime.sendMessage({ type: 'OPEN_UPGRADE_PAGE' });
      prompt.remove();
    });

    // Auto-remove after 30 seconds
    setTimeout(() => {
      if (document.getElementById('autohdpro-upgrade-prompt')) {
        prompt.remove();
      }
    }, 30000);
  }

  function showActivationToast(message, type = 'success') {
    try {
      const existing = document.getElementById('autohdpro-activation-toast');
      if (existing) {
        existing.remove();
      }

      const toast = document.createElement('div');
      toast.id = 'autohdpro-activation-toast';
      toast.style.cssText = `
        position: fixed;
        top: 24px;
        right: 24px;
        padding: 14px 18px;
        border-radius: 10px;
        background: ${type === 'error' ? '#E74C3C' : '#1ABC9C'};
        color: white;
        font-family: system-ui, -apple-system, sans-serif;
        font-size: 14px;
        z-index: 999999;
        box-shadow: 0 10px 30px rgba(0,0,0,0.25);
        max-width: 320px;
      `;

      toast.textContent = message;
      document.body.appendChild(toast);

      setTimeout(() => {
        toast.remove();
      }, 6000);
    } catch (error) {
      logger.error('Failed to display activation toast:', error);
    }
  }

  // Show usage notification when getting close to limit
  function showUsageNotification(usage) {
    const existingNotif = document.getElementById('autohdpro-usage-notif');
    if (existingNotif) {
      existingNotif.remove();
    }

    if (!usage || usage.unlimited || typeof usage.remaining !== 'number' || typeof usage.limit !== 'number') {
      return;
    }

    const notif = document.createElement('div');
    notif.id = 'autohdpro-usage-notif';
    notif.style.cssText = `
      position: fixed;
      bottom: 20px;
      right: 20px;
      background: #1a1a2e;
      color: white;
      padding: 16px 20px;
      border-radius: 8px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.2);
      z-index: 999998;
      font-family: system-ui, -apple-system, sans-serif;
      font-size: 14px;
      display: flex;
      align-items: center;
      gap: 12px;
      animation: fadeIn 0.3s ease-out;
    `;

    const percentage = Math.round((usage.conversions / usage.limit) * 100);
    const color = usage.remaining <= 10 ? '#ff4757' : usage.remaining <= 50 ? '#ffa502' : '#00d2d3';

    notif.innerHTML = `
      <style>
        @keyframes fadeIn {
          from { opacity: 0; transform: translateY(20px); }
          to { opacity: 1; transform: translateY(0); }
        }
      </style>
      <div style="width: 40px; height: 40px; border-radius: 50%; background: ${color}20; display: flex; align-items: center; justify-content: center;">
        <span style="font-weight: bold; color: ${color};">${usage.remaining}</span>
      </div>
      <div>
        <div style="font-weight: 500; margin-bottom: 2px;">HD Conversions Remaining</div>
        <div style="font-size: 12px; opacity: 0.8;">${percentage}% of monthly limit used</div>
      </div>
    `;

    document.body.appendChild(notif);

    // Auto-remove after 5 seconds
    setTimeout(() => {
      if (document.getElementById('autohdpro-usage-notif')) {
        notif.remove();
      }
    }, 5000);
  }

  // Inject page-context scripts if needed
  if (currentDriver && (currentDriver === window.YouTubeDriver || currentDriver === window.NetflixDriver)) {
    AutoHDProDOM.injectScriptFile('src/content/inject/site-hooks.js');
  }

  logger.info('AutoHDPro initialized successfully');

  // Removed periodic rescan - let mutation observer handle dynamic content
  // This was causing CPU overload

})();
