/**
 * AutoHDPro Storage for Service Worker
 * Service worker compatible storage management
 */

const DEFAULT_SETTINGS = {
  enabled: true,
  globalCaptionsOff: true,
  autoQuality: true,
  preferredQuality: 'highest', // highest, 2160p, 1440p, 1080p, 720p
  siteOverrides: {},
  debugMode: false,
  statistics: {
    videosEnhanced: 0,
    totalTimeSaved: 0
  },
  siteStats: {},
  subscription: {
    plan: 'free', // free or pro
    monthlyConversions: 0,
    monthlyLimit: 1000,
    resetDate: null,
    userId: null,
    totalConversions: 0,
    stripeCustomerId: null,
    status: 'inactive',
    currentPeriodEnd: null,
    lastStatusCheck: null
  }
};

const SITE_OVERRIDE_DEFAULTS = {
  enabled: true,
  captionsOff: true,
  quality: 'highest'
};

class ServiceWorkerStorage {
  constructor() {
    this.cache = null;
    this.listeners = new Set();
  }

  /**
   * Initialize storage with defaults
   */
  async init() {
    try {
      const stored = await chrome.storage.local.get('settings');
      
      if (!stored.settings) {
        await chrome.storage.local.set({ settings: DEFAULT_SETTINGS });
        this.cache = { ...DEFAULT_SETTINGS };
      } else {
        // Merge with defaults to ensure all keys exist
        this.cache = {
          ...DEFAULT_SETTINGS,
          ...stored.settings,
          subscription: {
            ...DEFAULT_SETTINGS.subscription,
            ...stored.settings?.subscription
          }
        };
      }

      this._normalizeSubscriptionDefaults();
      
      // Listen for changes from other contexts
      chrome.storage.onChanged.addListener(this._handleStorageChange.bind(this));
      
      return this.cache;
    } catch (error) {
      console.error('AutoHDPro: Storage initialization failed:', error);
      this.cache = { ...DEFAULT_SETTINGS };
      return this.cache;
    }
  }

  /**
   * Get all settings
   */
  async getSettings() {
    if (!this.cache) {
      await this.init();
    }
    return { ...this.cache };
  }

  /**
   * Get specific setting
   */
  async get(key) {
    const settings = await this.getSettings();
    return key.includes('.') 
      ? this._getNestedValue(settings, key)
      : settings[key];
  }

  /**
   * Update settings
   */
  async set(updates) {
    const settings = await this.getSettings();
    
    // Handle nested updates
    Object.entries(updates).forEach(([key, value]) => {
      if (key.includes('.')) {
        this._setNestedValue(settings, key, value);
      } else {
        settings[key] = value;
      }
    });
    
    this.cache = settings;
    this._normalizeSubscriptionDefaults();
    
    try {
      await chrome.storage.local.set({ settings });
    } catch (error) {
      console.error('AutoHDPro: Failed to save settings:', error);
    }
    
    this._notifyListeners(updates);
    return settings;
  }

  /**
   * Get site-specific override
   */
  async getSiteOverride(hostname) {
    const settings = await this.getSettings();
    const overrides = settings.siteOverrides || {};
    
    // Check exact match first
    if (overrides[hostname]) {
      return { ...SITE_OVERRIDE_DEFAULTS, ...overrides[hostname] };
    }
    
    // Check wildcard matches
    for (const [pattern, override] of Object.entries(overrides)) {
      if (pattern.includes('*')) {
        const regex = new RegExp(pattern.replace(/\*/g, '.*'));
        if (regex.test(hostname)) {
          return { ...SITE_OVERRIDE_DEFAULTS, ...override };
        }
      }
    }
    
    return null;
  }

  /**
   * Set site-specific override
   */
  async setSiteOverride(hostname, override) {
    const settings = await this.getSettings();
    
    if (!settings.siteOverrides) {
      settings.siteOverrides = {};
    }
    
    if (override === null) {
      // Remove override
      delete settings.siteOverrides[hostname];
    } else {
      settings.siteOverrides[hostname] = {
        ...SITE_OVERRIDE_DEFAULTS,
        ...override
      };
    }
    
    await this.set({ siteOverrides: settings.siteOverrides });
  }

  /**
   * Update statistics
   */
  async updateStats(stats) {
    const settings = await this.getSettings();
    const currentStats = settings.statistics || {};

    Object.entries(stats).forEach(([key, value]) => {
      if (typeof value === 'number') {
        currentStats[key] = (currentStats[key] || 0) + value;
      } else {
        currentStats[key] = value;
      }
    });

    const updatedSettings = await this.set({ statistics: currentStats });
    return updatedSettings.statistics;
  }

  /**
   * Get video count for a specific site
   */
  async getSiteVideoCount(hostname) {
    if (!hostname) {
      return 0;
    }

    const settings = await this.getSettings();
    const siteStats = settings.siteStats || {};
    return Number(siteStats[hostname]) || 0;
  }

  /**
   * Update video count for a specific site
   */
  async updateSiteVideoCount(hostname, increment = 1) {
    if (!hostname) {
      return 0;
    }

    const delta = Number(increment) || 0;
    if (!delta) {
      return this.getSiteVideoCount(hostname);
    }

    const settings = await this.getSettings();
    const siteStats = { ...(settings.siteStats || {}) };
    const current = Number(siteStats[hostname]) || 0;
    siteStats[hostname] = current + delta;

    const updatedSettings = await this.set({ siteStats });
    return Number(updatedSettings.siteStats?.[hostname]) || 0;
  }

  /**
   * Reset video count for a specific site
   */
  async resetSiteVideoCount(hostname) {
    if (!hostname) {
      return 0;
    }

    const settings = await this.getSettings();
    const siteStats = { ...(settings.siteStats || {}) };
    siteStats[hostname] = 0;

    const updatedSettings = await this.set({ siteStats });
    return Number(updatedSettings.siteStats?.[hostname]) || 0;
  }

  /**
   * Check and update usage for HD conversions
   */
  async checkUsage() {
    const settings = await this.getSettings();
    const subscription = settings.subscription || {};

    let dirty = false;

    // Reset monthly count if needed
    const now = new Date();
    const resetDate = subscription.resetDate ? new Date(subscription.resetDate) : null;

    if (!resetDate || now > resetDate) {
      // Set next reset date to first of next month
      const nextReset = new Date(now.getFullYear(), now.getMonth() + 1, 1);
      subscription.resetDate = nextReset.toISOString();
      subscription.monthlyConversions = 0;
      dirty = true;
    }

    if (typeof subscription.totalConversions !== 'number') {
      subscription.totalConversions = Number(subscription.totalConversions) || 0;
      dirty = true;
    }

    if (dirty) {
      await this.set({ subscription });
    }

    const isUnlimitedPlan = subscription.plan === 'pro';
    const rawLimit = Number(subscription.monthlyLimit);
    const effectiveLimit = Number.isFinite(rawLimit) ? rawLimit : (isUnlimitedPlan ? null : DEFAULT_SETTINGS.subscription.monthlyLimit);

    // Check if user has exceeded limit (only for free plan)
    if (!isUnlimitedPlan && typeof effectiveLimit === 'number' && subscription.monthlyConversions >= effectiveLimit) {
      return {
        allowed: false,
        remaining: 0,
        limit: effectiveLimit,
        conversions: subscription.monthlyConversions,
        totalConversions: subscription.totalConversions,
        plan: subscription.plan,
        resetDate: subscription.resetDate,
        unlimited: false
      };
    }

    return {
      allowed: true,
      remaining: typeof effectiveLimit === 'number'
        ? Math.max(effectiveLimit - subscription.monthlyConversions, 0)
        : null,
      limit: typeof effectiveLimit === 'number' ? effectiveLimit : null,
      conversions: subscription.monthlyConversions,
      totalConversions: subscription.totalConversions,
      plan: subscription.plan,
      resetDate: subscription.resetDate,
      unlimited: isUnlimitedPlan || typeof effectiveLimit !== 'number'
    };
  }

  /**
   * Increment HD conversion count
   */
  async incrementUsage() {
    const settings = await this.getSettings();
    const subscription = settings.subscription || {};

    // First check and reset if needed
    await this.checkUsage();

    // Increment the counter
    subscription.monthlyConversions = (subscription.monthlyConversions || 0) + 1;
    subscription.totalConversions = (subscription.totalConversions || 0) + 1;

    await this.set({ subscription });

    return subscription.monthlyConversions;
  }

  /**
   * Update subscription plan
   */
  async updateSubscription(plan, userId = null, metadata = {}) {
    const settings = await this.getSettings();
    const subscription = settings.subscription || {};

    subscription.plan = plan;
    subscription.userId = userId;
    subscription.status = metadata.status || (plan === 'pro' ? 'active' : 'inactive');
    subscription.currentPeriodEnd = metadata.currentPeriodEnd || null;
    subscription.lastStatusCheck = metadata.lastStatusCheck || subscription.lastStatusCheck || null;
    subscription.stripeCustomerId = metadata.stripeCustomerId || subscription.stripeCustomerId || null;

    // Pro plan has no limits
    if (plan === 'pro') {
      subscription.monthlyLimit = metadata.monthlyLimit ?? null;
    } else if (Number.isFinite(metadata.monthlyLimit)) {
      subscription.monthlyLimit = metadata.monthlyLimit;
    } else if (!Number.isFinite(subscription.monthlyLimit) || subscription.monthlyLimit === Infinity) {
      subscription.monthlyLimit = DEFAULT_SETTINGS.subscription.monthlyLimit;
    }

    await this.set({ subscription });
    return subscription;
  }

  /**
   * Reset to defaults
   */
  async reset() {
    this.cache = { ...DEFAULT_SETTINGS };
    this._normalizeSubscriptionDefaults();
    await chrome.storage.local.set({ settings: DEFAULT_SETTINGS });
    this._notifyListeners({ reset: true });
  }

  /**
   * Add change listener
   */
  onChange(callback) {
    this.listeners.add(callback);
    return () => this.listeners.delete(callback);
  }

  /**
   * Private: Handle storage changes
   */
  _handleStorageChange(changes, area) {
    if (area === 'local' && changes.settings) {
      this.cache = changes.settings.newValue;
      this._normalizeSubscriptionDefaults();
      this._notifyListeners({ external: true, ...this.cache });
    }
  }

  /**
   * Private: Notify listeners
   */
  _notifyListeners(changes) {
    this.listeners.forEach(callback => {
      try {
        callback(changes);
      } catch (e) {
        console.error('AutoHDPro: Storage listener error:', e);
      }
    });
  }

  /**
   * Private: Get nested value
   */
  _getNestedValue(obj, path) {
    return path.split('.').reduce((current, key) => current?.[key], obj);
  }

  /**
   * Private: Set nested value
   */
  _setNestedValue(obj, path, value) {
    const keys = path.split('.');
    const lastKey = keys.pop();
    const target = keys.reduce((current, key) => {
      if (!current[key]) current[key] = {};
      return current[key];
    }, obj);
    target[lastKey] = value;
  }

  /**
   * Private: Ensure subscription object contains default keys
   */
  _normalizeSubscriptionDefaults() {
    if (!this.cache) {
      return;
    }

    const subscription = this.cache.subscription || {};

    this.cache.subscription = {
      ...DEFAULT_SETTINGS.subscription,
      ...subscription,
      totalConversions: Number(subscription.totalConversions) || 0,
    };

    const limit = this.cache.subscription.monthlyLimit;
    if (typeof limit === 'string') {
      const parsed = Number(limit);
      this.cache.subscription.monthlyLimit = Number.isFinite(parsed) ? parsed : null;
    }

    if (this.cache.subscription.plan === 'pro' && !Number.isFinite(this.cache.subscription.monthlyLimit)) {
      this.cache.subscription.monthlyLimit = null;
    }

    if (this.cache.subscription.plan !== 'pro' && !Number.isFinite(this.cache.subscription.monthlyLimit)) {
      this.cache.subscription.monthlyLimit = DEFAULT_SETTINGS.subscription.monthlyLimit;
    }

    if (!this.cache.subscription.status) {
      this.cache.subscription.status = this.cache.subscription.plan === 'pro' ? 'active' : 'inactive';
    }

    if (typeof this.cache.subscription.lastStatusCheck !== 'string') {
      this.cache.subscription.lastStatusCheck = null;
    }
  }
}

// Create singleton instance for service worker
const storage = new ServiceWorkerStorage();

// Export for use in service worker
self.AutoHDProStorage = storage;
