/**
 * AutoHDPro Service Worker
 * Handles background tasks and extension lifecycle
 */

// Import service worker specific storage utilities and configuration
importScripts('config-sw.js', 'storage-sw.js');

const API_BASE_URL = self.AUTOHDPRO_SW_CONFIG?.apiBaseUrl || 'https://autohdpro.com';
const PRICING_URL = self.AUTOHDPRO_SW_CONFIG?.pricingUrl || `${API_BASE_URL.replace(/\/$/, '')}/pricing`;
const SUBSCRIPTION_STATUS_TTL = 1000 * 60 * 60 * 6; // 6 hours
const SUCCESS_BADGE_TIMEOUT = 4000; // 4 seconds

// Extension state
let extensionState = {
  enabled: true,
  tabStates: new Map()
};

const successBadgeTimers = new Map();

// Ensure badge state reflects persisted settings when the service worker starts
(async () => {
  try {
    const settings = await AutoHDProStorage.getSettings();
    await setEnabledBadge(settings.enabled !== false);
  } catch (_) {
    await setEnabledBadge(true);
  }
})();

// Initialize on install
chrome.runtime.onInstalled.addListener(async (details) => {
  console.log('AutoHDPro: Extension installed', details);
  
  // Initialize storage with defaults
  await AutoHDProStorage.init();
  
  // Set initial badge
  const initialSettings = await AutoHDProStorage.getSettings();
  await setEnabledBadge(initialSettings.enabled !== false);
  
  // Show welcome page on first install
  if (details.reason === 'install') {
    chrome.tabs.create({
      url: chrome.runtime.getURL('src/ui/welcome.html')
    });
  }
});

// Handle extension icon click (when popup is not set)
chrome.action.onClicked.addListener(async (tab) => {
  // Toggle enabled state
  const settings = await AutoHDProStorage.getSettings();
  const newEnabled = !settings.enabled;
  
  const updatedSettings = await AutoHDProStorage.set({ enabled: newEnabled });
  await setEnabledBadge(updatedSettings.enabled);
  
  // Notify content script
  try {
    await chrome.tabs.sendMessage(tab.id, {
      type: 'TOGGLE_ENABLED',
      enabled: newEnabled
    });
  } catch (e) {
    // Content script not loaded, inject it
    await injectContentScript(tab.id);
  }
});

// Handle messages from content scripts and popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  handleMessage(message, sender, sendResponse);
  return true; // Indicate async response
});

// Handle tab updates
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.status === 'loading') {
    await clearSuccessBadge(tabId);
  }

  if (changeInfo.status === 'complete' && tab.url) {
    // Check if we should inject on this tab
    const shouldInject = await shouldInjectOnUrl(tab.url);
    
    if (shouldInject) {
      try {
        const url = new URL(tab.url);
        const hostname = url.hostname;
        
        // Get existing site video count
        const siteVideoCount = await AutoHDProStorage.getSiteVideoCount(hostname);
        
        // Update tab state with preserved site count
        extensionState.tabStates.set(tabId, {
          url: tab.url,
          hostname: hostname,
          injected: false,
          videosFound: siteVideoCount
        });
        
        await restoreBadgeForTab(tabId);
        chrome.runtime.sendMessage({
          type: 'TAB_STATS_UPDATED',
          tabId,
          videosFound: siteVideoCount
        }).catch(() => {});
        
        // Try to inject if needed
        await injectContentScript(tabId);
      } catch (e) {
        // Fallback to old behavior if URL parsing fails
        extensionState.tabStates.set(tabId, {
          url: tab.url,
          injected: false,
          videosFound: 0
        });
        await restoreBadgeForTab(tabId);
        chrome.runtime.sendMessage({
          type: 'TAB_STATS_UPDATED',
          tabId,
          videosFound: 0
        }).catch(() => {});
        await injectContentScript(tabId);
      }
    } else if (extensionState.tabStates.has(tabId)) {
      extensionState.tabStates.delete(tabId);
      await restoreBadgeForTab(tabId);
      chrome.runtime.sendMessage({
        type: 'TAB_STATS_UPDATED',
        tabId,
        videosFound: 0
      }).catch(() => {});
    }
  }
});

// Handle tab removal
chrome.tabs.onRemoved.addListener((tabId) => {
  extensionState.tabStates.delete(tabId);
  clearSuccessBadge(tabId).catch(() => {});
});

// Handle main frame navigation, which is more reliable for reloads
chrome.webNavigation.onCommitted.addListener(async (details) => {
  // Only act on the main frame
  if (details.frameId !== 0) {
    return;
  }

  const { tabId, url } = details;
  const shouldInject = await shouldInjectOnUrl(url);

  if (shouldInject) {
    // Reset the state for the tab to ensure scripts re-run
    let hostname = null;
    let siteVideoCount = 0;
    try {
      const urlObj = new URL(url);
      hostname = urlObj.hostname;
      if (hostname) {
        siteVideoCount = await AutoHDProStorage.getSiteVideoCount(hostname);
      }
    } catch (_) {
      hostname = null;
      siteVideoCount = 0;
    }

    extensionState.tabStates.set(tabId, {
      url,
      hostname,
      injected: false,
      videosFound: siteVideoCount,
    });
    await restoreBadgeForTab(tabId);
    if (hostname) {
      chrome.runtime.sendMessage({
        type: 'TAB_STATS_UPDATED',
        tabId,
        videosFound: siteVideoCount
      }).catch(() => {});
    }
    await injectContentScript(tabId);
  } else if (extensionState.tabStates.has(tabId)) {
    extensionState.tabStates.delete(tabId);
  }
});

// Update badge based on enabled state
async function setEnabledBadge(enabled) {
  extensionState.enabled = Boolean(enabled);

  const defaultColor = extensionState.enabled ? '#1D4ED8' : '#F44336';
  const defaultText = extensionState.enabled ? '' : 'OFF';

  try {
    await chrome.action.setBadgeBackgroundColor({ color: defaultColor });
    await chrome.action.setBadgeText({ text: defaultText });
  } catch (error) {
    console.debug('AutoHDPro: Unable to set default badge state', error);
  }

  await clearAllSuccessBadges();
  await refreshTabBadges();
}

function formatBadgeCount(count) {
  if (!Number.isFinite(count) || count < 0) {
    return '';
  }

  if (count > 99) {
    return '99+';
  }

  return String(count);
}

async function updateTabBadge(tabId, count) {
  if (!extensionState.enabled) {
    return;
  }

  const text = formatBadgeCount(count);
  const successActive = successBadgeTimers.has(tabId);

  try {
    if (!successActive) {
      await chrome.action.setBadgeBackgroundColor({ tabId, color: '#1D4ED8' });
    }
    await chrome.action.setBadgeText({ tabId, text });
  } catch (error) {
    console.debug('AutoHDPro: Unable to update badge for tab', tabId, error);
  }
}

async function restoreBadgeForTab(tabId) {
  if (typeof tabId !== 'number') {
    return;
  }

  if (!extensionState.enabled) {
    try {
      await chrome.action.setBadgeBackgroundColor({ tabId, color: '#F44336' });
      await chrome.action.setBadgeText({ tabId, text: 'OFF' });
    } catch (error) {
      console.debug('AutoHDPro: Unable to restore OFF badge for tab', tabId, error);
    }
    return;
  }

  const count = extensionState.tabStates.get(tabId)?.videosFound || 0;
  await updateTabBadge(tabId, count);
}

async function refreshTabBadges() {
  const updates = Array.from(extensionState.tabStates.keys()).map((tabId) => restoreBadgeForTab(tabId));
  await Promise.all(updates);
}

async function clearAllSuccessBadges() {
  const entries = Array.from(successBadgeTimers.entries());
  successBadgeTimers.clear();

  for (const [, timer] of entries) {
    clearTimeout(timer);
  }

  await Promise.all(entries.map(([tabId]) => restoreBadgeForTab(tabId)));
}

async function clearSuccessBadge(tabId) {
  if (typeof tabId !== 'number') {
    return;
  }

  const existingTimer = successBadgeTimers.get(tabId);
  if (existingTimer) {
    clearTimeout(existingTimer);
    successBadgeTimers.delete(tabId);
  }

  await restoreBadgeForTab(tabId);
}

async function showSuccessBadge(tabId) {
  if (typeof tabId !== 'number' || !extensionState.enabled) {
    return;
  }

  await clearSuccessBadge(tabId);

  const count = extensionState.tabStates.get(tabId)?.videosFound || 0;
  const text = formatBadgeCount(count);

  try {
    await chrome.action.setBadgeBackgroundColor({ tabId, color: '#22C55E' });
    await chrome.action.setBadgeText({ tabId, text });
  } catch (error) {
    console.debug('AutoHDPro: Unable to show success badge for tab', tabId, error);
    return;
  }

  const timer = setTimeout(() => {
    successBadgeTimers.delete(tabId);
    restoreBadgeForTab(tabId).catch(() => {});
  }, SUCCESS_BADGE_TIMEOUT);

  successBadgeTimers.set(tabId, timer);
}

// Check if we should inject on a URL
function isChromeErrorPage(url) {
  if (!url) {
    return false;
  }

  if (url.startsWith('chrome-error://')) {
    return true;
  }

  if (url === 'about:blank') {
    return true;
  }

  return false;
}

async function shouldInjectOnUrl(url) {
  try {
    if (isChromeErrorPage(url)) {
      return false;
    }

    const urlObj = new URL(url);
    
    // Skip chrome:// and extension pages
    const blockedProtocols = new Set([
      'chrome:',
      'chrome-extension:',
      'about:',
      'edge:',
      'brave:',
      'safari-extension:',
      'chrome-error:'
    ]);

    if (blockedProtocols.has(urlObj.protocol)) {
      return false;
    }

    if (urlObj.hostname === 'chromewebdata') {
      return false;
    }
    
    // Check if extension is enabled
    const settings = await AutoHDProStorage.getSettings();
    if (!settings.enabled) {
      return false;
    }
    
    // Check for site override
    const override = await AutoHDProStorage.getSiteOverride(urlObj.hostname);
    if (override && !override.enabled) {
      return false;
    }
    
    return true;
  } catch (e) {
    return false;
  }
}

// Inject content script into tab
async function injectContentScript(tabId) {
  try {
    // Check if already injected
    const tabState = extensionState.tabStates.get(tabId);
    if (tabState?.injected) {
      return;
    }
    
    // Get tab info
    const tab = await chrome.tabs.get(tabId);
    if (!tab.url || !await shouldInjectOnUrl(tab.url)) {
      return;
    }
    
    // Inject scripts in order
    const scripts = [
      'src/utils/log.js',
      'src/utils/dom.js',
      'src/utils/storage.js',
      'src/content/engine/detector.js',
      'src/content/engine/quality.js',
      'src/content/engine/captions.js',
      'src/content/drivers/youtube.js',
      'src/content/drivers/netflix.js',
      'src/content/drivers/twitch.js',
      'src/content/drivers/vimeo.js',
      'src/content/drivers/dailymotion.js',
      'src/content/drivers/pornhub.js',
      'src/content/drivers/xhamster.js',
      'src/content/drivers/xvideos.js',
      'src/content/features/shorts-blocker.js',
      'src/content/content.js'
    ];
    
    for (const script of scripts) {
      await chrome.scripting.executeScript({
        target: { tabId },
        files: [script]
      });
    }
    
    // Mark as injected
    if (extensionState.tabStates.has(tabId)) {
      extensionState.tabStates.get(tabId).injected = true;
    }
    
    console.log(`AutoHDPro: Content scripts injected into tab ${tabId}`);
  } catch (error) {
    if (error?.message && error.message.includes('error page')) {
      console.warn('AutoHDPro: Skipping content script injection because the tab is showing an error page', {
        tabId,
        error: error.message
      });
      return;
    }

    console.error('AutoHDPro: Failed to inject content script:', error);
  }
}

async function handleEnabledChange(enabled) {
  await setEnabledBadge(enabled);

  if (!enabled) {
    return;
  }

  try {
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!activeTab?.id || !activeTab.url) {
      return;
    }

    if (!await shouldInjectOnUrl(activeTab.url)) {
      return;
    }

    const existingState = extensionState.tabStates.get(activeTab.id);
    extensionState.tabStates.set(activeTab.id, {
      ...existingState,
      url: activeTab.url,
      injected: existingState?.injected || false,
      videosFound: existingState?.videosFound ?? 0,
      lastConversion: existingState?.lastConversion
    });

    if (existingState?.injected) {
      try {
        await chrome.tabs.sendMessage(activeTab.id, { type: 'REFRESH_SETTINGS' });
      } catch (_) {}
      return;
    }

    await injectContentScript(activeTab.id);

    try {
      await chrome.tabs.sendMessage(activeTab.id, { type: 'REFRESH_SETTINGS' });
    } catch (_) {}
  } catch (error) {
    console.error('AutoHDPro: Failed to enable enhancements on active tab:', error);
  }
}

// Handle messages
async function handleMessage(message, sender, sendResponse) {
  try {
    switch (message.type) {
      case 'GET_SETTINGS':
        const settings = await AutoHDProStorage.getSettings();
        extensionState.enabled = Boolean(settings.enabled);
        sendResponse({ success: true, settings });
        break;
        
      case 'UPDATE_SETTINGS':
        const updated = await AutoHDProStorage.set(message.updates);
        sendResponse({ success: true, settings: updated });
        
        // Sync active tab when the global enabled state changes
        if ('enabled' in message.updates) {
          await handleEnabledChange(message.updates.enabled);
        }
        break;
        
      case 'GET_SITE_OVERRIDE':
        const override = await AutoHDProStorage.getSiteOverride(message.hostname);
        sendResponse({ success: true, override });
        break;
        
      case 'SET_SITE_OVERRIDE':
        await AutoHDProStorage.setSiteOverride(message.hostname, message.override);
        sendResponse({ success: true });
        break;
        
      case 'UPDATE_STATS':
        await AutoHDProStorage.updateStats(message.stats);
        sendResponse({ success: true });
        break;
        
      case 'VIDEO_DETECTED':
        // Update tab state and per-site count
        if (sender.tab?.id) {
          const tabState = extensionState.tabStates.get(sender.tab.id) || {};
          tabState.videosFound = (tabState.videosFound || 0) + 1;
          extensionState.tabStates.set(sender.tab.id, tabState);

          // Update per-site count if we have hostname
          if (tabState.hostname) {
            await AutoHDProStorage.updateSiteVideoCount(tabState.hostname, 1);
          }

          await updateTabBadge(sender.tab.id, tabState.videosFound);
          chrome.runtime.sendMessage({
            type: 'TAB_STATS_UPDATED',
            tabId: sender.tab.id,
            videosFound: tabState.videosFound
          }).catch(() => {});
        }

        // Update stats
        const updatedStats = await AutoHDProStorage.updateStats({
          videosEnhanced: 1
        });

        const totalVideosEnhanced = Number(updatedStats?.videosEnhanced);
        if (Number.isFinite(totalVideosEnhanced)) {
          console.info(`AutoHDPro: Videos enhanced: ${totalVideosEnhanced}`);
        } else {
          console.info('AutoHDPro: Video enhanced');
        }

        sendResponse({
          success: true,
          videosEnhanced: Number.isFinite(totalVideosEnhanced) ? totalVideosEnhanced : undefined
        });
        break;

      case 'GET_TAB_STATS':
        if (typeof message.tabId === 'number') {
          const tabStats = extensionState.tabStates.get(message.tabId);
          sendResponse({
            success: true,
            videosFound: Number(tabStats?.videosFound) || 0
          });
        } else {
          sendResponse({ success: false, videosFound: 0 });
        }
        break;

      case 'VIDEO_ENHANCEMENT_STARTED':
        if (sender.tab?.id) {
          await clearSuccessBadge(sender.tab.id);
        }
        sendResponse({ success: true });
        break;

      case 'VIDEO_ENHANCEMENT_SUCCESS':
        if (sender.tab?.id) {
          await showSuccessBadge(sender.tab.id);
        }
        sendResponse({ success: true });
        break;

      case 'VIDEO_ENHANCEMENT_FAILED':
        if (sender.tab?.id) {
          await clearSuccessBadge(sender.tab.id);
        }
        sendResponse({ success: true });
        break;

      case 'CHECK_USAGE':
        await ensureSubscriptionStatus();
        const usage = await AutoHDProStorage.checkUsage();
        sendResponse({ success: true, usage });
        break;

      case 'HD_CONVERSION_ATTEMPT':
        await ensureSubscriptionStatus();
        // Check if user can perform HD conversion
        const usageCheck = await AutoHDProStorage.checkUsage();

        if (!usageCheck.allowed) {
          // Show upgrade prompt
          const limitMessageValue = typeof usageCheck.limit === 'number' ? usageCheck.limit : 1000;
          sendResponse({
            success: false,
            needsUpgrade: true,
            usage: usageCheck,
            message: `You have reached your monthly limit of ${limitMessageValue} HD conversions. Upgrade to Pro for unlimited conversions!`
          });
        } else {
          // Rate limit: prevent more than 1 conversion per second from same tab
          const tabId = sender.tab?.id;
          if (tabId) {
            const now = Date.now();
            const lastConversion = extensionState.tabStates.get(tabId)?.lastConversion || 0;

            if (now - lastConversion < 1000) {
              // Too fast, likely a duplicate
              sendResponse({
                success: true,
                usage: usageCheck,
                duplicate: true
              });
              break;
            }

            // Update last conversion time
            const tabState = extensionState.tabStates.get(tabId) || {};
            tabState.lastConversion = now;
            extensionState.tabStates.set(tabId, tabState);
          }

          // Increment usage and allow conversion
          await AutoHDProStorage.incrementUsage();
          sendResponse({
            success: true,
            usage: {
              ...usageCheck,
              conversions: usageCheck.conversions + 1,
              totalConversions: (usageCheck.totalConversions || 0) + 1,
              remaining: usageCheck.remaining === null ? null : usageCheck.remaining - 1
            }
          });
        }
        break;

      case 'OPEN_UPGRADE_PAGE':
        chrome.tabs.create({
          url: PRICING_URL
        });
        sendResponse({ success: true });
        break;

      case 'ACTIVATE_PRO_SUBSCRIPTION':
        if (!message.sessionId) {
          sendResponse({ success: false, error: 'Missing Stripe session ID.' });
          break;
        }

        try {
          const response = await fetch(`${API_BASE_URL.replace(/\/$/, '')}/api/stripe/checkout-session?session_id=${encodeURIComponent(message.sessionId)}`);

          if (!response.ok) {
            const errorBody = await response.json().catch(() => ({}));
            throw new Error(errorBody.error || 'Subscription not ready yet.');
          }

          const data = await response.json();

          if (data.subscriptionStatus !== 'active' && data.status !== 'complete') {
            throw new Error('Subscription is not active yet. Please wait a moment and try again.');
          }

          await AutoHDProStorage.updateSubscription('pro', data.customerEmail || null, {
            stripeCustomerId: data.customerId || null,
            status: data.subscriptionStatus || 'active',
            currentPeriodEnd: data.currentPeriodEnd || null,
            lastStatusCheck: new Date().toISOString()
          });

          sendResponse({
            success: true,
            plan: 'pro',
            customerEmail: data.customerEmail || null,
            stripeCustomerId: data.customerId || null,
          });
        } catch (error) {
          console.error('AutoHDPro: Failed to activate subscription:', error);
          sendResponse({
            success: false,
            error: error instanceof Error ? error.message : 'Activation failed. Please contact support.',
          });
        }
        break;
        
      case 'GET_TAB_STATE':
        const tabId = sender.tab?.id || message.tabId;
        const state = extensionState.tabStates.get(tabId);
        sendResponse({ success: true, state });
        break;
        
      case 'INJECT_SCRIPT':
        // Inject a script into the page context
        if (sender.tab?.id) {
          await chrome.scripting.executeScript({
            target: { tabId: sender.tab.id },
            func: (code) => {
              const script = document.createElement('script');
              script.textContent = code;
              (document.head || document.documentElement).appendChild(script);
              script.remove();
            },
            args: [message.code]
          });
        }
        sendResponse({ success: true });
        break;
        
      case 'OPEN_OPTIONS':
        chrome.runtime.openOptionsPage();
        sendResponse({ success: true });
        break;

      case 'RE_CHECK_PAGE':
        if (sender.tab?.id) {
          console.log(`AutoHDPro: Re-checking tab ${sender.tab.id} for new videos.`);
          await injectContentScript(sender.tab.id);
        }
        sendResponse({ success: true });
        break;
        
      default:
        sendResponse({ success: false, error: 'Unknown message type' });
    }
  } catch (error) {
    console.error('AutoHDPro: Message handler error:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Periodic cleanup of old tab states
setInterval(() => {
  // Clean up tab states for tabs that no longer exist
  chrome.tabs.query({}, (tabs) => {
    const existingTabIds = new Set(tabs.map(t => t.id));
    
    for (const [tabId] of extensionState.tabStates) {
      if (!existingTabIds.has(tabId)) {
        extensionState.tabStates.delete(tabId);
      }
    }
  });
}, 60000); // Every minute

console.log('AutoHDPro: Service worker loaded');

// Ensure subscription status is refreshed on startup
ensureSubscriptionStatus({ force: true }).catch((error) => {
  console.error('AutoHDPro: Initial subscription status check failed:', error);
});

// Periodically refresh subscription status while worker is alive
setInterval(() => {
  ensureSubscriptionStatus().catch((error) => {
    console.error('AutoHDPro: Background subscription status refresh failed:', error);
  });
}, SUBSCRIPTION_STATUS_TTL);

async function ensureSubscriptionStatus({ force = false } = {}) {
  try {
    const subscription = await AutoHDProStorage.get('subscription');
    const plan = subscription?.plan;

    if (!subscription || (plan !== 'pro' && !force)) {
      return;
    }

    const lastChecked = subscription.lastStatusCheck ? new Date(subscription.lastStatusCheck).getTime() : 0;
    if (!force && lastChecked && Date.now() - lastChecked < SUBSCRIPTION_STATUS_TTL) {
      return;
    }

    const customerEmail = subscription.userId || null;
    const stripeCustomerId = subscription.stripeCustomerId || null;

    if (!customerEmail && !stripeCustomerId) {
      return;
    }

    const response = await fetch(`${API_BASE_URL.replace(/\/$/, '')}/api/stripe/subscription-status`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        customerEmail,
        stripeCustomerId
      })
    });

    if (!response.ok) {
      throw new Error(`Subscription status request failed: ${response.status}`);
    }

    const data = await response.json();
    const isActive = Boolean(data.active);
    const nextPlan = isActive ? 'pro' : 'free';

    const metadata = {
      stripeCustomerId: data.stripeCustomerId || stripeCustomerId,
      status: data.status || (isActive ? 'active' : 'inactive'),
      currentPeriodEnd: data.currentPeriodEnd || null,
      lastStatusCheck: new Date().toISOString()
    };

    if (Number.isFinite(data.limit)) {
      metadata.monthlyLimit = data.limit;
    }

    await AutoHDProStorage.updateSubscription(nextPlan, customerEmail, metadata);
  } catch (error) {
    console.error('AutoHDPro: Subscription status check failed:', error);
    const now = new Date().toISOString();
    await AutoHDProStorage.set({ 'subscription.lastStatusCheck': now });
  }
}
